package org.tkit.rhpam.quarkus.domain.models;

import lombok.Getter;
import lombok.Setter;

import javax.enterprise.inject.Instance;
import javax.enterprise.inject.spi.CDI;
import javax.persistence.*;
import java.io.Serializable;
import java.security.Principal;
import java.util.Date;
import java.util.Objects;
import java.util.UUID;

/**
 * The Failed step entity representing issue in the process(business or technical)
 */
@Getter
@Setter
@Entity
@Table(name = "PL_FAILED_STEP",
        indexes = {
                @Index(name = "FS_PROCESS_INST_ID", columnList = "PROCESS_INST_ID"),
                @Index(name = "FS_PROCESS_STEP_LOG_GUID", columnList = "PS_LOG_GUID")
        }
)
public class FailedStep implements Serializable {
    
    /**
     * String ID of entity
     */
    @Id
    @Column(name = "GUID")
    private String id = UUID.randomUUID().toString();
    
    @Column(name = "ORIGINAL_MESSAGE", length = 5000)
    private String originalJMSMessage;
    
    @Column(name = "HEADERS", length = 5000)
    private String headers;
    
    @Column(name = "PS_LOG_GUID")
    private String processStepLogGuid;
    
    @Column(name = "PROCESS_NAME")
    private String processName;
    
    @Column(name = "PROCESS_STEP_NAME")
    private String processStepName;
    
    @Column(name = "PROCESS_INST_ID")
    private Long processInstanceId;
    
    @Column(name = "ERROR_CODE")
    private String errorCode;
    
    @Column(name = "ERROR_MSG")
    private String errorMessage;
    
    @Column(name = "WORK_ITEM_ID")
    private Long workItemId;
    
    @Column(name = "REF_BID")
    private String referenceBid;
    
    @Column(name = "REF_KEY")
    private String referenceKey;
    
    @Column(name = "STATUS")
    @Enumerated(EnumType.STRING)
    private FailedStepStatus status;
    
    @Column(name = "EXECUTION_COUNT")
    private Integer executionCount;
    
    @Column(name = "EXTERNAL_ISSUE_ID")
    private String externalIssueId;
    
    @Column(name = "TYPE")
    @Enumerated(EnumType.STRING)
    private FailureType failureType;
    
    @Column(name = "RESOLUTION_OPTIONS")
    private String resolutionOptionsString;
    
    @Column(name = "RESOLUTION_SIGNAL")
    private String resolutionSignalReceiver;
    
    @Column(name = "RESOLUTION")
    private String resolution;
    
    @Column(name = "RESOLUTION_DATE")
    @Temporal(TemporalType.TIMESTAMP)
    private Date resolutionDate;
    
    @Column(name = "RESOLUTION_ORIGIN")
    private String resolutionOrigin;
    
    @Column(name = "DEEP_LINK")
    private String deepLink;
    
    @Column(name = "DEPLOYMENT_ID")
    private String deploymentId;
    
    @Column(name = "PROCESS_ID")
    private String processId;
    @Column(name = "NODE_ID")
    
    private String nodeId;
    @Column(name = "NODE_NAME")
    
    private String nodeName;
    @Column(name = "PARENT_PROCESS_INSTANCE_ID")
    private String parentProcessInstanceId;
    
    @Version
    @Column(name = "OPTLOCK", nullable = false)
    private Integer version;
    
    private Date creationDate;
    
    private Date modificationDate;
    
    /**
     * The Process Step Status
     */
    public enum FailedStepStatus {
        
        /**
         * OPEN status, when the failed step is created or the same error occurs the x-time.
         */
        OPEN,
        
        /**
         * The step was retried.
         */
        RETRY,
        
        /**
         * The step was skipped.
         */
        SKIP,
        /**
         * Closed failed step status.
         */
        CLOSED
    }
    
    /**
     * The Step Failure Type
     */
    public enum FailureType {
        
        /**
         * TECHNICAL_ERROR.
         */
        TECHNICAL_ERROR,
        
        /**
         * BUSINESS_ERROR.
         */
        BUSINESS_ERROR
    }
    
    /**
     * {@inheritDoc }
     *
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        FailedStep other = (FailedStep) obj;
        Object guid = getId();
        Object otherGuid = other.getId();
        
        if (guid == null) {
            if (otherGuid != null) {
                return false;
            } else {
                return super.equals(obj);
            }
        } else if (!guid.equals(otherGuid)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * {@inheritDoc }
     *
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + Objects.hashCode(getId());
        return result;
    }
    
    @PrePersist
    public void prePersist() {
        Date date = new Date();
        setCreationDate(date);
        setModificationDate(date);
    }
    
    @PreUpdate
    public void preUpdate() {
        setModificationDate(new Date());
    }
}
