package org.tkit.rhpam.quarkus.domain.models;

import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

import javax.persistence.*;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Domain process info is a domain-process synchronization object, that contains information about current and historical process executions for given domain entity.
 * For every domain object there is at most 1 DomainProcessInfo - its ID = DOMAIN_ENTITY_TYPE#DOMAIN_ENTITY_ID.
 * The entity is update using explicit DB locking during the execution of business process.
 * For every process and subprocess, an entry in the {@link #stageFlags} will be created, with process Id being the key. The stage flag is update on process start and end, or in case of error.
 */
@Entity
@Getter
@Setter
@Table(name = "pl_domain_process_info")
@ToString(of = {"id", "currentProcessId", "currentProcessStatus", "currentProcessStepName", "currentProcessStepStatus"})
public class DomainProcessInfo {
    @Id
    private String id;
    private boolean processActive;
    private String currentProcessId;
    private Long currentProcessInstanceId;
    @Temporal(TemporalType.TIMESTAMP)
    private Date currentProcessStartTime;
    @Enumerated(EnumType.STRING)
    private ProcessStatus currentProcessStatus;
    private String currentProcessInitiator;
    private String currentProcessStepName;
    private String currentProcessStepStatus;
    @ElementCollection(fetch = FetchType.LAZY)
    @MapKeyColumn(name = "stage")
    @CollectionTable(
            name = "pl_process_stages",
            joinColumns = @JoinColumn(name = "process_info_id")
    )
    private Map<String, StageFlag> stageFlags = new HashMap<>();

    /**
     * The current process step time.
     */
    @Temporal(TemporalType.TIMESTAMP)
    private Date currentProcessStepTime;

    /**
     * Instantiates a new Domain process info.
     */
    public DomainProcessInfo() {

    }

    /**
     * Instantiates a new Domain process info.
     *
     * @param domainRefKey the domain ref key
     * @param domainRefId  the domain ref id
     */
    public DomainProcessInfo(String domainRefKey, String domainRefId) {
        this.id = domainRefKey + "#" + domainRefId;
    }

    /**
     * The enum Process status.
     */
    public enum ProcessStatus {
        /**
         * Aborted process status.
         */
        ABORTED,
        /**
         * Pending process status.
         */
        PENDING,
        /**
         * Running process status.
         */
        RUNNING,
        /**
         * Completed process status.
         */
        COMPLETED,
        /**
         * Error process status.
         */
        ERROR;
    }

}
