package org.tkit.rhpam.quarkus.messaging;

import ext.api.centrallog.api.ProcessLogEventEmitter;
import ext.api.centrallog.model.*;
import io.smallrye.reactive.messaging.amqp.IncomingAmqpMetadata;
import io.vertx.core.json.JsonObject;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.eclipse.microprofile.opentracing.Traced;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.tkit.rhpam.quarkus.common.MessageKeys;
import org.tkit.rhpam.quarkus.domain.daos.FailedStepDAO;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.domain.models.FailedStepSearchCriteria;
import org.tkit.rhpam.quarkus.messaging.common.Constants;
import org.tkit.rhpam.quarkus.messaging.common.MessageUtil;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecutionResult;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import java.io.IOException;
import java.util.*;


/**
 * The type Failed step service.
 */
@ApplicationScoped
@Slf4j
@Traced
public class FailedStepService {
    private static final String DEEP_LINK = "deepLink";
    private static final String DEPLOYMENT_ID = "deploymentId";
    private static final String REFERENCE_KEY = "referenceKey";
    private static final String REFERENCE_BID = "referenceBid";
    private static final String PROCESS_STEP_NAME = "processStepName";

    @Inject
    ProcessLogEventEmitter centralLogEmitter;
    @Inject
    FailedStepDAO service;
    @ConfigProperty(name = "quarkus.resteasy.path")
    Optional<String> contextPath;
    @ConfigProperty(name = "quarkus.application.name")
    Optional<String> quarkusAppName;
    @ConfigProperty(name = "tkit.service.name")
    Optional<String> tkitServiceName;
    @ConfigProperty(name = "org.tkit.rhpam.support.tool.callback.url")
    Optional<String> supportCallbackUrl;

    public FailedStep createOrUpdateFailedStep(Message<String> message) throws RhpamException {
        try {
            FailedStep failedStep = getExistingFailedStep(message);
            if (failedStep != null) {
                // a failed step for this error already exists so update the status
                failedStep.setStatus(FailedStep.FailedStepStatus.OPEN);
                failedStep.setExecutionCount(failedStep.getExecutionCount() + 1);
                failedStep = service.update(failedStep);
            } else {
                failedStep = constructFailedStepFromMessage(message);
                failedStep = service.create(failedStep);
            }

            return failedStep;

        } catch (Exception e) {
            throw new IllegalStateException(FailedStepServiceErrorKeys.FAILED_TO_CREATE_FAILED_STEP.name(), e);
        }
    }

    public void emitFailedStepEvent(FailedStep failedStep) throws RhpamException {
        OpenIncidentEvent event = new OpenIncidentEvent();
        event.setErrorCode(failedStep.getErrorCode());
        event.setErrorMessage(failedStep.getErrorMessage());
        event.setErrorType(ErrorType.fromValue(failedStep.getFailureType().name()));
        event.setNodeId(failedStep.getNodeId());
        event.setExternalIssueId(failedStep.getExternalIssueId());
        event.setPriority(PriorityLevel.HIGH);
        List<ResolutionOption> options = new ArrayList<>();
        String[] optionArr = failedStep.getResolutionOptionsString().split(",");
        for (String item : optionArr) {
            options.add(new ResolutionOption().key(item).label(item).callBackUrl(constructResolutionCallbackUrlForCentralLog(failedStep.getId())));
        }
        event.setResolutionOptions(options);
        event.setRetryCount(failedStep.getExecutionCount());
        event.setCorrelationId(failedStep.getId());
        event.setEventTime(failedStep.getModificationDate());
        event.setIncidentId(failedStep.getId());
        event.setProcessEventType(EventType.OPEN_INCIDENT_EVENT);
        event.setProcessId(failedStep.getProcessId());
        event.setProcessInstanceId(String.valueOf(failedStep.getProcessInstanceId()));
        event.setExecutionId(String.valueOf(failedStep.getWorkItemId()));
        event.getMetadata().put(DEEP_LINK, failedStep.getDeepLink());
        event.getMetadata().put(DEPLOYMENT_ID, failedStep.getDeploymentId());
        event.getMetadata().put(REFERENCE_KEY, failedStep.getReferenceKey());
        event.getMetadata().put(REFERENCE_BID, failedStep.getReferenceBid());
        event.getMetadata().put(PROCESS_STEP_NAME, failedStep.getProcessStepName());
        centralLogEmitter.emitProcessEvent(event);
    }

    public void emitResolvedFailedStepEvent(FailedStep failedStep) throws RhpamException {
        ResolvedIncidentEvent resolvedIncidentEvent = new ResolvedIncidentEvent();
        resolvedIncidentEvent.setProcessInstanceId(failedStep.getProcessInstanceId().toString());
        resolvedIncidentEvent.setProcessId(failedStep.getProcessId());
        resolvedIncidentEvent.setProcessEventType(EventType.RESOLVED_INCIDENT_EVENT);
        resolvedIncidentEvent.setIncidentId(failedStep.getId());
        resolvedIncidentEvent.setEventTime(new Date());
        resolvedIncidentEvent.setExecutionId(failedStep.getWorkItemId().toString());
        resolvedIncidentEvent.setCorrelationId(failedStep.getId());
        resolvedIncidentEvent.setNodeId(failedStep.getNodeId());
        resolvedIncidentEvent.getMetadata().put(DEEP_LINK, failedStep.getDeepLink());
        resolvedIncidentEvent.getMetadata().put(DEPLOYMENT_ID, failedStep.getDeploymentId());
        resolvedIncidentEvent.getMetadata().put(REFERENCE_KEY, failedStep.getReferenceKey());
        resolvedIncidentEvent.getMetadata().put(REFERENCE_BID, failedStep.getReferenceBid());
        resolvedIncidentEvent.getMetadata().put(PROCESS_STEP_NAME, failedStep.getProcessStepName());
        resolvedIncidentEvent.setResolutionKey(failedStep.getResolution());
        resolvedIncidentEvent.setResolutionSource(tkitServiceName.orElse(quarkusAppName.orElse("UNKNOWN")));
        resolvedIncidentEvent.setResolutionStatus(ResolutionStatus.RESOLVED);
        centralLogEmitter.emitProcessEvent(resolvedIncidentEvent);
    }

    /**
     * Create Failed step for a process step that was technically executed ok, but resulted in business error
     *
     * @param workItem        process execution data
     * @param executionResult process execution result data
     * @return created failed step
     * @throws IOException in case of error
     */
    public FailedStep createOrUpdateBusinessFailedStep(ProcessStepExecution workItem, ProcessStepExecutionResult executionResult) throws IOException {
        FailedStep failedStep = null;
        FailedStepSearchCriteria criteria = new FailedStepSearchCriteria();
        criteria.setFailureType(FailedStep.FailureType.BUSINESS_ERROR);
        criteria.setProcessStepName(workItem.getName());
        criteria.setProcessInstanceId(workItem.getProcessInstanceId());
        List<FailedStep> failedStepList = service.findBySearchCriteria(criteria);
        // a failed step for this error already exists so update the status
        if (failedStepList != null && failedStepList.size() == 1) {
            failedStep = failedStepList.get(0);
            failedStep.setExecutionCount(failedStep.getExecutionCount() + 1);
            failedStep.setStatus(FailedStep.FailedStepStatus.OPEN);
            failedStep.setWorkItemId(workItem.getWorkItemId());
            failedStep.setNodeId(workItem.getNodeId());
            return service.update(failedStep);
        } else {
            failedStep = new FailedStep();
            failedStep.setFailureType(FailedStep.FailureType.BUSINESS_ERROR);

            //failedStep.setResolutionOptionsString(resolutionOptions);
            //failedStep.setResolutionSignalReceiver(resolutionSignalId);
            failedStep.setExecutionCount(1);
            failedStep.setProcessInstanceId(workItem.getProcessInstanceId());

            if (executionResult.getParameters().get(Constants.FAILED_STEP_ERROR_CODE) != null) {
                failedStep.setErrorCode((String) executionResult.getParameters().get(Constants.FAILED_STEP_ERROR_CODE));
            } else {
                log.warn(MessageKeys.FAILED_STEP_WITHOUT_ERROR_CODE.toString());
            }
            if (executionResult.getParameters().get(Constants.FAILED_STEP_ERROR_MSG) != null) {
                failedStep.setErrorMessage((String) executionResult.getParameters().get(Constants.FAILED_STEP_ERROR_MSG));
            }
            if (executionResult.getParameters().get(Constants.FAILED_STEP_DEEP_LINK) != null) {
                failedStep.setDeepLink((String) executionResult.getParameters().get(Constants.FAILED_STEP_DEEP_LINK));
            }
            failedStep.setNodeId(workItem.getNodeId());
            failedStep.setNodeName(workItem.getName());
            failedStep.setProcessId(workItem.getProcessId());
            failedStep.setProcessInstanceId(workItem.getProcessInstanceId());
            failedStep.setParentProcessInstanceId(workItem.getParentProcessInstanceId() != null ?
                    String.valueOf(workItem.getParentProcessInstanceId()) : null);
            failedStep.setProcessStepName(workItem.getName());
            failedStep.setProcessName(workItem.getProcessName());
            failedStep.setStatus(FailedStep.FailedStepStatus.OPEN);
            failedStep.setWorkItemId(workItem.getWorkItemId());
            failedStep.setReferenceBid(workItem.getReferenceId());
            failedStep.setReferenceKey(workItem.getReferenceKey());
            failedStep.setDeploymentId(workItem.getDeploymentId());
            //TODO hack, we misuses the field, to store the output params, because we do not have other option
            failedStep.setOriginalJMSMessage(MessageUtil.serializeBody(executionResult.getParameters()));
            return service.create(failedStep);
        }
    }

    public FailedStep updateFailedStep(ProcessStepExecution workItem, String referencedStepName, String resolutionOptions,
                                       String resolutionSignalId) throws RhpamException {
        FailedStep failedStep = null;
        FailedStepSearchCriteria criteria = new FailedStepSearchCriteria();
        criteria.setFailureType(FailedStep.FailureType.BUSINESS_ERROR);
        criteria.setProcessStepName(referencedStepName);
        criteria.setStatus(FailedStep.FailedStepStatus.OPEN);
        criteria.setProcessInstanceId(workItem.getProcessInstanceId());
        List<FailedStep> failedStepList = service.findBySearchCriteria(criteria);
        // a failed step for this error already exists so update the status
        if (failedStepList != null && failedStepList.size() > 0) {
            log.warn("More then 1 Open failed step found for : {}", referencedStepName);
            failedStep = failedStepList.get(0);
        } else {
            throw new RhpamException(MessageKeys.BUSINESS_ERR_HANDLER_MISSING_FAILED_STEP.getErrorCode(), MessageKeys.BUSINESS_ERR_HANDLER_MISSING_FAILED_STEP.formatMessage(referencedStepName), null);
        }
        failedStep.setResolutionOptionsString(resolutionOptions);
        failedStep.setResolutionSignalReceiver(resolutionSignalId);
        failedStep = service.update(failedStep);

        return failedStep;
    }

    /**
     * Try to find the FailedStep object
     *
     * @param msg failed step Message
     * @return found FailedStep based on workItemId or null with updated status
     */
    private FailedStep getExistingFailedStep(Message msg) {
        FailedStep failedStep = null;

        try {

            Long workItemId = MessageUtil.getProperty(msg, MessageUtil.PROP_WORK_ITEM_ID);

            FailedStepSearchCriteria criteria = new FailedStepSearchCriteria();
            criteria.setWorkItemId(workItemId);
            List<FailedStep> failedStepList = service.findBySearchCriteria(criteria);

        } catch (Exception e) {
            log.error("Error to load failed step object.", e);
        }

        return failedStep;
    }

    /**
     * Create the FailedStep object from the message
     *
     * @param msg failed step Message
     * @return FailedStep object
     */
    private FailedStep constructFailedStepFromMessage(Message<String> msg) {
        FailedStep failedStep = null;

        try {
            failedStep = new FailedStep();
            Long processInstanceId = MessageUtil.getProperty(msg, MessageUtil.PROP_PROCESS_INSTANCE_ID);
            Long workItemId = MessageUtil.getProperty(msg, MessageUtil.PROP_WORK_ITEM_ID);
            String processId = MessageUtil.getProperty(msg, MessageUtil.PROP_PROCESS_ID);
            Object referenceId = MessageUtil.getProperty(msg, MessageUtil.PROP_REFERENCE_BID);
            String referenceKey = MessageUtil.getProperty(msg, MessageUtil.PROP_REFERENCE_KEY);
            String processName = MessageUtil.getProperty(msg, MessageUtil.PROP_PROCESS_NAME);
            String nodeName = MessageUtil.getProperty(msg, MessageUtil.PROP_KIE_NODE_NAME);
            Long nodeId = MessageUtil.getProperty(msg, MessageUtil.PROP_KIE_NODE_ID);

            failedStep.setNodeId(String.valueOf(nodeId));
            failedStep.setOriginalJMSMessage(msg.getPayload());
            failedStep.setHeaders(createHeadersText(msg));
            failedStep.setProcessInstanceId(processInstanceId);
            failedStep.setProcessId(processId);
            failedStep.setProcessName(processName);
            failedStep.setProcessStepName(nodeName);
            failedStep.setErrorCode(MessageUtil.getProperty(msg, MessageUtil.FAILED_STEP_ERROR_CODE));
            failedStep.setErrorMessage(MessageUtil.getProperty(msg, MessageUtil.FAILED_STEP_ERROR_MSG));
            failedStep.setDeepLink(MessageUtil.getProperty(msg, MessageUtil.FAILED_STEP_DEEP_LINK));
            failedStep.setReferenceBid("" + referenceId);
            failedStep.setReferenceKey(referenceKey);
            failedStep.setWorkItemId(workItemId);
            failedStep.setExecutionCount(0);
            failedStep.setResolutionOptionsString("RETRY");
            failedStep.setStatus(FailedStep.FailedStepStatus.OPEN);
            failedStep.setFailureType(FailedStep.FailureType.TECHNICAL_ERROR);
        } catch (Exception e) {
            log.error("Error creating the failed step object.", e);
        }

        return failedStep;
    }

    /**
     * Create Json text from all header parameters of a jms message
     *
     * @param message jms message
     * @return json string from the header parameters
     */
    private String createHeadersText(Message<String> message) {
        String headerText = null;
        Map<String, Object> result = new HashMap<>();

        try {

            JsonObject properties = message.getMetadata(IncomingAmqpMetadata.class).get().getProperties();

            result = properties.getMap();

            headerText = MessageUtil.serializeBody(result);
        } catch (Exception e) {
            log.error("Error serializing the header parameters of a message.", e);
        }

        return headerText;
    }

    private String constructResolutionCallbackUrlForCentralLog(String failedStepId) {
        StringBuilder urlBuilder = new StringBuilder();

        if (supportCallbackUrl.isPresent()) {
            urlBuilder.append(supportCallbackUrl.get());
        } else {
            String appName = tkitServiceName.orElse(quarkusAppName.orElse("UNKNOWN"));
            urlBuilder.append("http://");
            urlBuilder.append(appName);
        }
        urlBuilder.append(contextPath.orElse("/"));
        urlBuilder.append("/failedSteps/");
        urlBuilder.append(failedStepId);
        urlBuilder.append("/resolution");

        return urlBuilder.toString();
    }

    public enum FailedStepServiceErrorKeys {
        FAILED_TO_CREATE_FAILED_STEP
    }
}
