package org.tkit.rhpam.quarkus.messaging;

import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.opentracing.util.GlobalTracer;
import io.quarkus.arc.Arc;
import io.smallrye.reactive.messaging.amqp.AmqpMessage;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.context.ManagedExecutor;
import org.eclipse.microprofile.context.ThreadContext;
import org.eclipse.microprofile.reactive.messaging.Acknowledgment;
import org.eclipse.microprofile.reactive.messaging.Incoming;
import org.slf4j.MDC;
import org.tkit.quarkus.log.cdi.BusinessLoggingUtil;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.AdditionalErrorInfo;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecutionResult;
import org.tkit.rhpam.quarkus.process.BusinessProcess;
import org.tkit.rhpam.quarkus.process.ProcessStepService;
import org.tkit.rhpam.quarkus.tracing.TraceFromMessage;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import javax.transaction.UserTransaction;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

/**
 * The type Tkit rhpam wih service.
 */
@ApplicationScoped
@Slf4j
public class TkitRhpamWIHService {

    /**
     * The Process client service v 2.
     */
    @Inject
    ProcessClientServiceV2 processClientServiceV2;

    @Inject
    ThreadContext threadContext;
    @Inject
    ManagedExecutor managedExecutor;

    @Inject
    Tracer configuredTracer;

    @Inject
    BusinessErrorHandlerStepService businessErrorHandlerStepService;

    @Inject
    UserTransaction transaction;


    /**
     * Test 2 completion stage.
     *
     * @param message the message
     * @return the completion stage
     */
//    @Transactional(Transactional.TxType.REQUIRED)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("tkitrhpamwih")
    @TraceFromMessage
    public CompletionStage<Void> onWIHMessage(AmqpMessage<String> message) {
        Span span = configuredTracer.activeSpan().setOperationName("onWIHMessage");
        log.info("WIH attempt: {}, data: {}", message.getDeliveryCount(), message.getApplicationProperties());

        // TX 1 start
        if (log.isDebugEnabled()) {
            log.debug("Message properties: {}", message.getApplicationProperties());
        }
        return threadContext.withContextCapture(CompletableFuture.runAsync(() -> {
            try (Scope scope = configuredTracer.scopeManager().activate(span, false)) {
                // TX 2 start, TX 1 pause
                // if we fail here, whole execution fails. JMS message is returned to queue
                ProcessStepExecution execution;

                try {
                    execution = processClientServiceV2.msgToStepExecution(message);
                    setTracingMetadata(execution);
                    processClientServiceV2.startLog(execution);
                } catch (Exception e) {
                    message.nack(e);
                    return;
                }

                Arc.container().requestContext().activate();
                ProcessStepService stepService = Arc.container().instance(ProcessStepService.class,
                        new BusinessProcess.Literal(execution.getProcessId())).get();
                Arc.container().requestContext().deactivate();
                if (stepService == null) {
                    log.error("No ProcessStepService for {}", execution.getProcessId());
                    message.nack(new IllegalArgumentException("No ProcessStepService for " + execution.getProcessId()));
                    return;
                }

                try {
                    ProcessStepExecutionResult result = stepService.doExecute(execution);
                    handleStepSuccess(execution, result, message);
                    message.ack();
                } catch (Exception e) {
                    AdditionalErrorInfo additionalErrorInfo = stepService.provideErrorInformation(execution, e);
                    log.error("Business execution failed for step: {} with error: {}", execution.getName(), e.getMessage());
                    handleStepFailed(execution, e, message, additionalErrorInfo);
                }
            } finally {
                this.clearTracingMeta();
                // Remove all business information after you are done with step execution
                BusinessLoggingUtil.removeAll();
            }

        }, managedExecutor));

    }

    private void clearTracingMeta() {
        MDC.remove("TKIT_PROCESS_ID");
        MDC.remove("TKIT_PROCESS_INSTANCE_ID");
        MDC.remove("TKIT_PROCESS_STEP");
        MDC.remove("TKIT_PROCESS_REF");
        MDC.remove("TKIT_PROCESS_REF_TYPE");
    }

    private void setTracingMetadata(ProcessStepExecution execution) {
        MDC.put("TKIT_PROCESS_ID", execution.getProcessId());
        MDC.put("TKIT_PROCESS_INSTANCE_ID", String.valueOf(execution.getProcessInstanceId()));
        MDC.put("TKIT_PROCESS_STEP", String.valueOf(execution.getName()));
        MDC.put("TKIT_PROCESS_REF", execution.getReferenceId());
        MDC.put("TKIT_PROCESS_REF_TYPE", execution.getReferenceKey());
        // if there is a tracer with active span modify it's name according to current process step
        if (GlobalTracer.isRegistered()) {
            Tracer tracer = GlobalTracer.get();
            if (tracer != null && tracer.activeSpan() != null) {
                tracer.activeSpan().setOperationName(execution.getName());
                tracer.activeSpan().setTag("process", execution.getProcessName());
                tracer.activeSpan().setTag("processInstance", execution.getProcessInstanceId());
                tracer.activeSpan().setTag("processRef", execution.getReferenceId());
                tracer.activeSpan().setTag("processRefType", execution.getReferenceKey());
            }
        }
    }

    /**
     * Test 2 completion stage.
     *
     * @param message the message
     * @return the completion stage
     */
    @Transactional(Transactional.TxType.REQUIRES_NEW)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("businesserrorwih")
    @TraceFromMessage
    public CompletionStage<Void> onBusinessErrorWIHMessage(AmqpMessage<String> message) {

        Span span = configuredTracer.activeSpan().setOperationName("onBusinessErrorWIHMessage");
        log.info("BusinessErrorWIH data: {}", message.getApplicationProperties());
        // TX 1 start
        return threadContext.withContextCapture(CompletableFuture.runAsync(() -> {
            try (Scope scope = configuredTracer.scopeManager().activate(span, false)) {
                // TX 2 start, TX 1 pause
                // if we fail here, whole execution fails. JMS message is returned to queue
                ProcessStepExecution execution;
                try {
                    execution = processClientServiceV2.msgToStepExecution(message);
                    setTracingMetadata(execution);
                    processClientServiceV2.startLog(execution);
                } catch (Exception e) {
                    message.nack(e);
                    return;
                }
                // TX 2 end
                long startTime = System.currentTimeMillis();
                log.info("Process work item {} started. Attempt: {}", execution, message.getDeliveryCount() + 1);
                ProcessStepService stepService = businessErrorHandlerStepService;

                try {
                    ProcessStepExecutionResult result = stepService.doExecute(execution);
                    handleStepSuccess(execution, result, message);
                    message.ack();
                } catch (Exception e) {
                    AdditionalErrorInfo additionalErrorInfo = stepService.provideErrorInformation(execution, e);
                    log.error("Business execution failed for step: {}", execution.getName(), e);
                    handleStepFailed(execution, e, message, additionalErrorInfo);
                }
            } finally {
                this.clearTracingMeta();
                // Remove all business information after you are done with step execution
                BusinessLoggingUtil.removeAll();
            }

        }, managedExecutor));
    }

    private void handleStepFailed(ProcessStepExecution execution, Throwable t, AmqpMessage<String> message, AdditionalErrorInfo additionalErrorInfo) {
        try {
            processClientServiceV2.endLog(execution, null, message, t, additionalErrorInfo);
            //endLog will throw en exception if the message should be retried(nacked)
            //if it does not throw exception, then it create a new technical incident(new msg in FailedStepQueue)
            //and we then ack the orig message, because we want to pause the process now
            message.ack();
        } catch (RhpamException rhe) {
            message.nack(rhe);
        } catch (Exception e) {
            //we get unexpected error, but we wanted to retry the msg anyhow
            //this would most probably be a bug in endlog, or hard technical error
            log.error("Unexpected Error in endlog", e);
            message.nack(e);
        }

    }

    private void handleStepSuccess(ProcessStepExecution execution, ProcessStepExecutionResult processStepExecutionResult,
                                   AmqpMessage<String> message) {
        try {
            processClientServiceV2.endLog(execution, processStepExecutionResult, message, null, null);
        } catch (Exception e) {
            log.error("Error in endlog", e);
        }
    }
}