package org.tkit.rhpam.quarkus.process;

import lombok.extern.slf4j.Slf4j;
import org.tkit.rhpam.quarkus.domain.daos.FailedStepDAO;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.domain.models.FailedStepSearchCriteria;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.List;

@ApplicationScoped
@Transactional
@Slf4j
public class ResolveStepService {

    @Inject
    FailedStepDAO failedStepDAO;

    @Inject
    RetryStepService retryStepService;

    /**
     * Resolves business failed steps
     *
     * @param processInstanceId process instance id of the failed step
     * @param resolutionSource  name of the signal which resolves the failed step
     * @param processStepName   name of the failed step
     * @throws RhpamException
     */
    public void resolveBusinessFailedSteps(Long processInstanceId, String resolutionOption, String resolutionSource, String processStepName) throws Exception {
        List<FailedStep> failedSteps = findOpenBusinessFailedSteps(processInstanceId, processStepName);
        for (FailedStep failedStep : failedSteps) {
            retryStepService.resolveFailedStep(failedStep.getId(), resolutionOption, resolutionSource);
            log.info("Business failed steps connected with processId: {} and processInstanceId: {} were resolved successfully", failedStep.getProcessId(), processInstanceId);
        }
    }

    /**
     * Finds business failed steps by instanceId and stepName
     * @param processInstanceId process instance id of the failed step
     * @return list of failed steps
     */
    private List<FailedStep> findOpenBusinessFailedSteps(Long processInstanceId, String processStepName) {
        FailedStepSearchCriteria criteria = new FailedStepSearchCriteria();
        criteria.setProcessInstanceId(processInstanceId);
        criteria.setFailureType(FailedStep.FailureType.BUSINESS_ERROR);
        criteria.setStatus(FailedStep.FailedStepStatus.OPEN);
        criteria.setProcessStepName(processStepName);
        return failedStepDAO.findBySearchCriteria(criteria);
    }

}