/*
 * Copyright (c) 2018, 1000kit.org, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.tkit.rhpam.quarkus.process;

import io.jaegertracing.internal.JaegerSpanContext;
import io.netty.util.internal.StringUtil;
import io.opentracing.Tracer;
import io.smallrye.reactive.messaging.amqp.OutgoingAmqpMetadata;
import io.vertx.core.json.JsonObject;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.eclipse.microprofile.opentracing.Traced;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.eclipse.microprofile.reactive.messaging.Metadata;
import org.eclipse.microprofile.rest.client.inject.RestClient;
import org.tkit.quarkus.log.cdi.context.TkitLogContext;
import org.tkit.rhpam.quarkus.domain.daos.DomainProcessInfoDAO;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;
import org.tkit.rhpam.quarkus.messaging.duplicate.DuplicateDetectionIdConfig;
import org.tkit.rhpam.quarkus.messaging.duplicate.DuplicateDetectionIdGenerator;
import org.tkit.rhpam.quarkus.messaging.common.MessageUtil;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.emitters.JbpmMessageEmitter;
import org.tkit.rhpam.quarkus.messaging.model.CommandType;
import org.tkit.rhpam.quarkus.rs.clients.KieServerRestClient;
import org.tkit.rhpam.quarkus.tracing.TracingUtils;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import static org.tkit.rhpam.quarkus.messaging.common.MessageUtil.*;

/**
 * The process service.
 */
@Slf4j
@ApplicationScoped
@Transactional(Transactional.TxType.REQUIRED)
@Traced
public class ProcessService {

    @Inject
    DomainProcessInfoDAO domainProcessInfoDAO;

    @Inject
    JbpmMessageEmitter jbpmMessageEmitter;

    @Inject
    Tracer configuredTracer;

    @ConfigProperty(name = "kie.admin.user")
    private Optional<String> usernameParam;

    @ConfigProperty(name = "kie.admin.pwd")
    private Optional<String> passwordParam;


    @Inject
    @RestClient
    KieServerRestClient kieServerRestClient;

    /**
     * Starts the process.
     *
     * @param deploymentId the deployment ID.
     * @param processId    the ID of the process
     * @param referenceBid business reference bid to be defined in the step logs
     * @param referenceKey the reference key.
     * @param duplicateDetection the id used to detect if start action process is a duplicate, see: <a href="https://activemq.apache.org/components/artemis/documentation/latest/duplicate-detection.html">Duplicate Message Detection</a>
     * @param data         the input data for the process.
     * @return the process log GUID.
     * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_START_PROCESS}
     */
    public String startProcess(String deploymentId, String processId, Object referenceBid, String referenceKey, DuplicateDetectionIdConfig duplicateDetection, Map<String, Object> data) throws RhpamException {
        try {
            Map<String, Object> tracingMeta = new HashMap<>();
            String parameters = enhanceParameters(data, tracingMeta);

            updateDomainProcessInfo(processId, referenceBid, referenceKey);

            String guid = UUID.randomUUID().toString();
            String duplicateDetectionKey = guid;

            if (duplicateDetection != null) {
                duplicateDetectionKey = !StringUtil.isNullOrEmpty(duplicateDetection.getExplicitKey())
                        ? duplicateDetection.getExplicitKey()
                        : duplicateDetection.isUseDefaultAlgorithm()
                        ? DuplicateDetectionIdGenerator.get(deploymentId, processId, referenceBid, referenceKey)
                        : duplicateDetectionKey;
            }

            OutgoingAmqpMetadata meta = OutgoingAmqpMetadata.builder()
                    .withMessageId(UUID.randomUUID().toString())
                    .withCreationTime(new Date().getTime())
                    .withApplicationProperties(JsonObject.mapFrom(tracingMeta))
                    .withApplicationProperty(PROP_DEPLOYMENT_ID, deploymentId)
                    .withApplicationProperty(PROP_PROCESS_LOG_GUID, guid)
                    .withApplicationProperty(PROP_REFERENCE_BID, referenceBid)
                    .withApplicationProperty(PROP_REFERENCE_KEY, referenceKey)
                    .withApplicationProperty(PROP_PROCESS_ID, processId)
                    .withApplicationProperty(PROP_DUPLICATE_DETECTION_ID, duplicateDetectionKey)
                    .withApplicationProperty(PROP_CMD, CommandType.START_PROCESS.name())
                    .build();


            Message<String> toEmit = Message.of(parameters, Metadata.of(meta), () -> {
                // Called when the message is acknowledged.
                return CompletableFuture.completedFuture(null);
            });
            jbpmMessageEmitter.emit(toEmit);

            return guid;
        } catch (Exception ex) {
            throw new RhpamException(ErrorKeys.ERROR_START_PROCESS.name(), ex);
        }
    }

    /**
         * Starts the process.
         *
         * @param deploymentId the deployment ID.
         * @param processId    the ID of the process
         * @param referenceBid business reference bid to be defined in the step logs
         * @param referenceKey the reference key.
         * @param data         the input data for the process.
         * @return the process log GUID.
         * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_START_PROCESS}
         */
    public String startProcess(String deploymentId, String processId, Object referenceBid, String referenceKey, Map<String, Object> data) throws RhpamException {

        try {
            Map<String, Object> tracingMeta = new HashMap<>();
            String parameters = enhanceParameters(data, tracingMeta);

            updateDomainProcessInfo(processId, referenceBid, referenceKey);

            String guid = UUID.randomUUID().toString();
            OutgoingAmqpMetadata meta = OutgoingAmqpMetadata.builder()
                    .withMessageId(UUID.randomUUID().toString())
                    .withCreationTime(new Date().getTime())
                    .withApplicationProperties(JsonObject.mapFrom(tracingMeta))
                    .withApplicationProperty(PROP_DEPLOYMENT_ID, deploymentId)
                    .withApplicationProperty(PROP_PROCESS_LOG_GUID, guid)
                    .withApplicationProperty(PROP_REFERENCE_BID, referenceBid)
                    .withApplicationProperty(PROP_REFERENCE_KEY, referenceKey)
                    .withApplicationProperty(PROP_PROCESS_ID, processId)
                    .withApplicationProperty(PROP_CMD, CommandType.START_PROCESS.name())
                    .build();


            Message<String> toEmit = Message.of(parameters, Metadata.of(meta), () -> {
                // Called when the message is acknowledged.
                return CompletableFuture.completedFuture(null);
            });
            jbpmMessageEmitter.emit(toEmit);

            return guid;
        } catch (Exception ex) {
            throw new RhpamException(ErrorKeys.ERROR_START_PROCESS.name(), ex);
        }
    }

    /**
     * Starts the process.
     *
     * @param deploymentId the deployment ID.
     * @param processId    the ID of the process
     * @param referenceBid business reference bid to be defined in the step logs
     * @param referenceKey the reference key.
     * @param data         the input data for the process.
     * @return the process log GUID.
     * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_START_PROCESS}
     */
    public long startProcessSync(String deploymentId, String processId, Object referenceBid, String referenceKey, Map<String, Object> data) throws RhpamException {

        try {
            Map<String, Object> tracingMeta = new HashMap<>();
            enhanceParameters(data, tracingMeta);

            updateDomainProcessInfo(processId, referenceBid, referenceKey);

            String guid = UUID.randomUUID().toString();
            data.put(PROP_REFERENCE_KEY, referenceKey);
            data.put(PROP_REFERENCE_BID, referenceBid);
            data.put(PROP_DEPLOYMENT_ID, deploymentId);
            data.put(PROP_PROCESS_ID, processId);
            data.put(PROP_PROCESS_LOG_GUID, guid);
            data.putAll(tracingMeta);

            String username = usernameParam.map(Object::toString).orElseThrow(() -> new RhpamException("Kie server username can not be null. " +
                    "Please provide property kie.admin.user with correct value", Response.Status.BAD_REQUEST));
            String password = passwordParam.map(Object::toString).orElseThrow(() -> new RhpamException("Kie server password can not be null." +
                    "Please provide property kie.admin.pwd with correct value", Response.Status.BAD_REQUEST));
            String authentication = getClientCredentialsAuth(username, password);
            return kieServerRestClient.startProcessInstance(authentication, deploymentId, processId, data);
        } catch (WebApplicationException wae) {
            String err = wae.getResponse().readEntity(String.class);
            log.error("Error response from kieserver : {} body {}", wae.getResponse().getStatus(), err);
            throw new RhpamException(ErrorKeys.ERROR_START_PROCESS.name(), wae);
        } catch (Exception ex) {
            throw new RhpamException(ErrorKeys.ERROR_START_PROCESS.name(), ex);
        }
    }

    private String enhanceParameters(Map<String, Object> data, Map<String, Object> tracingMeta) throws IOException {
        if (configuredTracer != null) {
            TracingUtils.inject(tracingMeta, configuredTracer.activeSpan(), configuredTracer);
            JaegerSpanContext spanContext = (JaegerSpanContext) configuredTracer.activeSpan().context();
            data.put("TRACE_ID", spanContext.getTraceId());
        }
        if (TkitLogContext.get() != null) {
            data.put(PROP_X_CORRELATION_ID, TkitLogContext.get().correlationId);
        }
        return MessageUtil.serializeBody(data);
    }

    private String getClientCredentialsAuth(String username, String password) {
        String tmp = username + ":" + password;
        return "Basic " + Base64.getEncoder().encodeToString(tmp.getBytes(StandardCharsets.UTF_8));
    }

    /**
     * Sends the message to the process.
     *
     * @param deploymentId      the deployment ID.
     * @param processInstanceId the process instance ID.
     * @param messageId         the message ID.
     * @param data              the input data.
     * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_SEND_SIGNAL}
     */
    public void sendMessage(String deploymentId, Long processInstanceId, String messageId, Object data) throws RhpamException {
        sendSignal(deploymentId, processInstanceId, "Message-" + messageId, data);
    }

    /**
     * Sends the signal to the process.
     *
     * @param deploymentId      the deployment ID.
     * @param processInstanceId the process instance ID.
     * @param signalId          the signal ID.
     * @param data              the input data.
     * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_SEND_SIGNAL}
     */
    public void sendSignal(String deploymentId, Long processInstanceId, String signalId, Object data) throws RhpamException {

        try {
            Map<String, Object> tmp = new HashMap<>();
            if (data != null) {
                tmp.put(MessageUtil.PROP_SIGNAL_DATA, data);
            }
            String parameters = MessageUtil.serializeBody(tmp);

            Message<String> toEmit = Message.of(parameters);
            OutgoingAmqpMetadata meta = OutgoingAmqpMetadata.builder()
                    .withApplicationProperty(PROP_DEPLOYMENT_ID, deploymentId)
                    .withApplicationProperty(PROP_PROCESS_INSTANCE_ID, processInstanceId)
                    .withApplicationProperty(PROP_SIGNAL_ID, signalId)
                    .withApplicationProperty(PROP_CMD, CommandType.SEND_SIGNAL.name())
                    .build();
            toEmit = toEmit.addMetadata(meta);
            //TODO tracing
            jbpmMessageEmitter.emit(toEmit);

        } catch (Exception ex) {
            throw new RhpamException(ErrorKeys.ERROR_SEND_SIGNAL.name(), ex);
        }
    }

    private void updateDomainProcessInfo(String processId, Object referenceBid, String referenceKey) {
        DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(referenceKey, String.valueOf(referenceBid));
        if (dpi == null) {
            dpi = new DomainProcessInfo(referenceKey, String.valueOf(referenceBid));
        }
        dpi.setCurrentProcessId(processId);
        dpi.setCurrentProcessStartTime(new Date());
        dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.PENDING);
        dpi.setProcessActive(true);
        dpi.setCurrentProcessStepName(null);
        dpi.setCurrentProcessStepStatus(null);

        domainProcessInfoDAO.update(dpi);
    }


    enum ErrorKeys {

        ERROR_START_PROCESS,

        ERROR_SEND_SIGNAL
    }
}
