package org.tkit.rhpam.quarkus.process;

import org.tkit.rhpam.quarkus.common.MessageKeys;
import org.tkit.rhpam.quarkus.messaging.common.Constants;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.AdditionalErrorInfo;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecutionResult;

/**
 * The interface Process step service.
 */
public interface ProcessStepService {

    /**
     * Execute the process work item.
     * If this method results in an exception, it is treated as Technical error in Tkit error handling.
     * In case of exception, the tkit-rhpam will invoke {@link #provideErrorInformation(ProcessStepExecution, Exception)} method, where you can
     * provide error code, message and deeplink for better error analysis. If not provided and exception is of type {@link RhpamException} then its
     * getErrorCode() is used as failed step error code.
     * <p>
     * When you want to indicate a negative outcome which should be handled by the Business Error Handler WIH later in the process, specify these 3
     * variables in output:
     * {@link Constants#FAILED_STEP_ERROR_CODE} - for error code,
     * {@link Constants#FAILED_STEP_ERROR_MSG} - for error message text,
     * {@link Constants#FAILED_STEP_DEEP_LINK} - for backlink to the failed object, e.g. link to Object detail in our UI view,
     *
     * @param workItem the process work item.
     * @return the result parameters for the workflow.
     * @throws RhpamException if the method fails.
     */
    ProcessStepExecutionResult doExecute(ProcessStepExecution workItem) throws RhpamException;

    /**
     * Provide any additional error information in case the processing of <code>#doExecute</code> has failed. In case you dont want to provide any
     * additional info, return null.
     *
     * @param workItem workitem that failed
     * @param exc      exception that occurred
     * @return instance of {@link AdditionalErrorInfo}, most important attributes are <code>errorCode</code> and <code>deepLink</code> - for back
     * navigation from support tool.
     */
    default AdditionalErrorInfo provideErrorInformation(ProcessStepExecution workItem, Exception exc) {
       return AdditionalErrorInfo.createDefault(workItem, exc);
    }

    ;


}
