package org.tkit.rhpam.quarkus.domain.daos;

import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.opentracing.Traced;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;
import org.tkit.rhpam.quarkus.domain.models.StageFlag;

import javax.enterprise.context.ApplicationScoped;
import javax.persistence.EntityManager;
import javax.persistence.LockModeType;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;


/**
 * The type Domain process info dao.
 */
@ApplicationScoped
@Traced
@Transactional
@Slf4j
public class DomainProcessInfoDAO {
    @PersistenceContext
    private EntityManager em;

    @Transactional(Transactional.TxType.NOT_SUPPORTED)
    public DomainProcessInfo find(String domainRefKey, String domainRefId) {
        String id = domainRefKey + "#" + domainRefId;
        try {
            return em.createQuery("select dpi from DomainProcessInfo dpi left join fetch dpi.stageFlags where dpi.id = :id",
                            DomainProcessInfo.class)
                    .setParameter("id", id)
                    .getSingleResult();
        } catch (NoResultException nre) {
            return null;
        }
    }

    @Transactional(Transactional.TxType.NOT_SUPPORTED)
    public DomainProcessInfo find(Long processInstanceId) {
        try {
            return em.createQuery("select dpi from DomainProcessInfo dpi left join fetch dpi.stageFlags where dpi.currentProcessInstanceId = :processInstanceId",
                            DomainProcessInfo.class)
                    .setParameter("processInstanceId", processInstanceId)
                    .getSingleResult();
        } catch (NoResultException nre) {
            return null;
        }
    }

    @Transactional(Transactional.TxType.NOT_SUPPORTED)
    public List<DomainProcessInfo> find(List<Long> processInstanceIds) {
        if (processInstanceIds.size() > 1000 ) {
            log.warn("Fetching with big number of processInstanceIds {}, it might have a bad impact on performance", processInstanceIds.size());
        }
        return em.createQuery("select dpi from DomainProcessInfo dpi left join fetch dpi.stageFlags where dpi.currentProcessInstanceId in :processInstanceIds",
                        DomainProcessInfo.class)
                .setParameter("processInstanceIds", processInstanceIds)
                .getResultList();
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public DomainProcessInfo selectForUpdate(String domainRefKey, String domainRefId) {
        String id = domainRefKey + "#" + domainRefId;
        try {
            return em.createQuery("select dpi from DomainProcessInfo dpi left join fetch dpi.stageFlags where dpi.id = :id",
                    DomainProcessInfo.class)
                    .setParameter("id", id)
                    .setLockMode(LockModeType.PESSIMISTIC_WRITE)
                    .getSingleResult();
        } catch (NoResultException nre) {
            return null;
        }
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public DomainProcessInfo selectForUpdate(String domainRefKey, String domainRefId, String processId) {
        String id = domainRefKey + "#" + domainRefId;
        try {
            return em.createQuery("select dpi from DomainProcessInfo dpi left join fetch dpi.stageFlags where dpi.id = :id AND dpi.currentProcessId = :processId",
                    DomainProcessInfo.class)
                    .setParameter("id", id)
                    .setParameter("processId", processId)
                    .setLockMode(LockModeType.PESSIMISTIC_WRITE)
                    .getSingleResult();
        } catch (NoResultException nre) {
            return null;
        }
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public DomainProcessInfo selectForUpdate(Long processInstanceId) {
        try {
            return em.createQuery("select dpi from DomainProcessInfo dpi left join fetch dpi.stageFlags where dpi.currentProcessInstanceId = :processInstanceId",
                    DomainProcessInfo.class)
                    .setParameter("processInstanceId", processInstanceId)
                    .setLockMode(LockModeType.PESSIMISTIC_WRITE)
                    .getSingleResult();
        } catch (NoResultException nre) {
            return null;
        }
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public void delete(DomainProcessInfo domainProcessInfo) {
        em.remove(domainProcessInfo);
        em.flush();
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public DomainProcessInfo update(DomainProcessInfo toUpdate) {
        toUpdate = em.merge(toUpdate);
        em.flush();
        return toUpdate;
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public DomainProcessInfo addStageFlag(String domainRefKey, String domainRefId, String flagName, Date date, String info) {
        DomainProcessInfo dpi = this.selectForUpdate(domainRefKey, domainRefId);
        StageFlag stageFlag = new StageFlag();
        stageFlag.setDate(date);
        stageFlag.setInfo(info);
        dpi.getStageFlags().put(flagName, stageFlag);
        return this.update(dpi);
    }
}
