package org.tkit.rhpam.quarkus.domain.daos;

import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.opentracing.Traced;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessHistory;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;

import javax.enterprise.context.ApplicationScoped;
import javax.persistence.EntityManager;
import javax.persistence.LockModeType;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;

import static org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo.ProcessStatus.RUNNING;

/**
 * The Domain Process History DAO.
 */
@ApplicationScoped
@Traced
@Transactional
@Slf4j
public class DomainProcessHistoryDAO {
    @PersistenceContext
    private EntityManager em;

    @Transactional(Transactional.TxType.NOT_SUPPORTED)
    public DomainProcessHistory find(String instanceId) {
        try {
            return em.createQuery("SELECT dph FROM DomainProcessHistory dph WHERE dph.primaryKey.instanceId = :instanceId",
                            DomainProcessHistory.class)
                    .setParameter("instanceId", instanceId)
                    .getSingleResult();
        } catch (NoResultException nre) {
            return null;
        }
    }

    @Transactional(Transactional.TxType.NOT_SUPPORTED)
    public List<DomainProcessHistory> find(List<String> instanceIds) {
        if (instanceIds.size() > 1000 ) {
            log.warn("Fetching with big number of instanceIds {}, it might have a bad impact on performance", instanceIds.size());
        }
        return em.createQuery("SELECT dph FROM DomainProcessHistory dph WHERE dph.primaryKey.instanceId in :instanceIds",
                        DomainProcessHistory.class)
                .setParameter("instanceIds", instanceIds)
                .getResultList();
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public void logProcessStart(String domainRef, String processId, String instanceId) {
        DomainProcessHistory.PrimaryKey primaryKey = new DomainProcessHistory.PrimaryKey();
        primaryKey.setDomainRef(domainRef);
        primaryKey.setProcessId(processId);
        primaryKey.setInstanceId(instanceId);

        DomainProcessHistory domainProcessHistory = new DomainProcessHistory();
        domainProcessHistory.setPrimaryKey(primaryKey);
        domainProcessHistory.setStatus(RUNNING);
        domainProcessHistory.setStartTimestamp(new Date());

        em.persist(domainProcessHistory);
        em.flush();
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public void logProcessEnd(String domainRef, String processId, String instanceId, DomainProcessInfo.ProcessStatus processStatus) {
        try {
            DomainProcessHistory domainProcessHistory =
                em.createQuery("SELECT dph FROM DomainProcessHistory dph WHERE dph.primaryKey.domainRef = :domainRef and dph.primaryKey.processId = :processId and dph.primaryKey.instanceId = :instanceId", DomainProcessHistory.class)
                        .setParameter("domainRef", domainRef)
                        .setParameter("processId", processId)
                        .setParameter("instanceId", instanceId)
                        .setLockMode(LockModeType.PESSIMISTIC_WRITE)
                        .getSingleResult();

            domainProcessHistory.setStatus(processStatus);
            domainProcessHistory.setEndTimestamp(new Date());

            em.merge(domainProcessHistory);
            em.flush();
        } catch (NoResultException nre) {
            log.error("No Domain Process History was found for Process ID {}, Instance ID {}, Domain Reference {} on process end.",
                    processId, instanceId, domainRef);
        }
    }

    @Transactional(Transactional.TxType.REQUIRED)
    public List<DomainProcessHistory> find(String domainRefKey, String domainRefId, String processId, DomainProcessInfo.ProcessStatus status) {
        return em.createQuery("SELECT dph FROM DomainProcessHistory dph WHERE dph.primaryKey.domainRef = :domainRef and dph.primaryKey.processId = :processId and dph.status = :status", DomainProcessHistory.class)
                .setParameter("domainRef", domainRefKey + "#" + domainRefId)
                .setParameter("processId", processId)
                .setParameter("status", status)
                .getResultList();
    }
}
