package org.tkit.rhpam.listener;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.util.JSONPObject;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.jbpm.services.task.commands.TaskContext;
import org.kie.api.task.TaskEvent;
import org.kie.api.task.TaskLifeCycleEventListener;
import org.kie.api.task.model.OrganizationalEntity;
import org.kie.api.task.model.Task;
import org.kie.api.task.model.TaskData;
import org.kie.internal.task.api.model.InternalPeopleAssignments;
import org.kie.server.api.KieServerConstants;
import org.kie.server.api.KieServerEnvironment;
import org.mvel2.util.Make;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tkit.rhpam.task.model.*;

import java.io.IOException;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class UserTaskListener implements TaskLifeCycleEventListener {

    private Logger log = LoggerFactory.getLogger(UserTaskListener.class);
    /** URL for the user task list api call
     * EX: http://tkit-task-list.test.svc.cluster.local:8080/task-list-rs/tasks */
    private String apiUrl;

    /**
     * Own url for callback
     * EX: http://myapp-kieserver-1.rhpam.svc.cluster.local:8080
     */
    private String kieApiUrl;

    private String namespace;

    private String serverId;

    public UserTaskListener(String url, String namespace, String kieApiUrl) {
        this(url, namespace);
        this.kieApiUrl = kieApiUrl;
    }

    public UserTaskListener(String url, String namespace) {
        this(url);
        this.namespace = namespace;
    }

    public UserTaskListener(String url) {
        this.apiUrl = url;
        serverId = KieServerEnvironment.getServerId();
        if (serverId == null) {
            serverId = KieServerEnvironment.getServerName();
        }
    }

    @Override public void beforeTaskActivatedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskActivatedEvent {}", taskEvent.getTask(), taskEvent.getTaskContext());
    }

    @Override public void beforeTaskClaimedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskClaimedEvent {}", taskEvent);
    }

    @Override public void beforeTaskSkippedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskSkippedEvent {}", taskEvent);
    }

    @Override public void beforeTaskStartedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskStartedEvent {}", taskEvent);
    }

    @Override public void beforeTaskStoppedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskStoppedEvent {}", taskEvent);
    }

    @Override public void beforeTaskCompletedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskCompletedEvent {}", taskEvent);
    }

    @Override public void beforeTaskFailedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskFailedEvent {}", taskEvent);
    }

    @Override public void beforeTaskAddedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskAddedEvent {}", taskEvent.getTask(), taskEvent.getTaskContext());
    }

    @Override public void beforeTaskExitedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskExitedEvent {}", taskEvent);
    }

    @Override public void beforeTaskReleasedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskReleasedEvent {}", taskEvent);
    }

    @Override public void beforeTaskResumedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskResumedEvent {}", taskEvent);
    }

    @Override public void beforeTaskSuspendedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskSuspendedEvent {}", taskEvent);
    }

    @Override public void beforeTaskForwardedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskForwardedEvent {}", taskEvent);
    }

    @Override public void beforeTaskDelegatedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskDelegatedEvent {}", taskEvent);
    }

    @Override public void beforeTaskNominatedEvent(TaskEvent taskEvent) {
        log.info("beforeTaskNominatedEvent {}", taskEvent);
    }

    @Override public void afterTaskActivatedEvent(TaskEvent taskEvent) {
        log.info("afterTaskActivatedEvent {}", taskEvent);
    }

    @Override public void afterTaskClaimedEvent(TaskEvent taskEvent) {
        log.info("afterTaskClaimedEvent {}", taskEvent);
    }

    @Override public void afterTaskSkippedEvent(TaskEvent taskEvent) {
        log.info("afterTaskSkippedEvent {}", taskEvent);
    }

    @Override public void afterTaskStartedEvent(TaskEvent taskEvent) {
        log.info("afterTaskStartedEvent {}", taskEvent);
    }

    @Override public void afterTaskStoppedEvent(TaskEvent taskEvent) {
        log.info("afterTaskStoppedEvent {}", taskEvent);
    }

    @Override public void afterTaskCompletedEvent(TaskEvent taskEvent) {
        log.info("afterTaskCompletedEvent {}", taskEvent);

        Task task = taskEvent.getTask();
        if (task == null) {
            log.error("Task is empty");
            return;
        }

        if (task.getTaskData() == null) {
            log.warn("task data empty");
            return;
        }
        TaskData taskData = task.getTaskData();

        StringBuilder sb = new StringBuilder(apiUrl);
        sb.append("/").append(taskData.getDeploymentId());
        sb.append("/").append(taskData.getProcessInstanceId());
        sb.append("/").append(task.getId());
        sb.append("/complete");

        HttpPost request = new HttpPost(sb.toString());

        // add request headers
        request.addHeader(HttpHeaders.ACCEPT, "application/json");
        request.addHeader(HttpHeaders.CONTENT_TYPE, "application/json");

        TaskCompleteRequest completeRequest = new TaskCompleteRequest();
        completeRequest.setAuthor(taskEvent.getTaskContext().getUserId());
        completeRequest.setResolutionComment("Complete from jbpm server");
        completeRequest.setResolutionDate(new Date());

        Map<String, String> outputParams = taskData.getTaskOutputVariables().entrySet().stream()
            .collect(Collectors.toMap(Map.Entry::getKey, e -> String.valueOf(e.getValue())));
        completeRequest.setResolutionParameters(outputParams);

        ObjectMapper mapper = new ObjectMapper();
        try {
            request.setEntity(new StringEntity(mapper.writeValueAsString(completeRequest)));
        } catch (UnsupportedEncodingException e) {
            log.error("Error creating json request", e);
            return;
        } catch (JsonProcessingException e) {
            log.error("Error creating json request", e);
            return;
        }

        try (
            CloseableHttpClient httpClient = HttpClients.createDefault();
            CloseableHttpResponse response = httpClient.execute(request)) {

            // Get HttpResponse Status
            log.info(response.getStatusLine().toString());        // HTTP/1.1 200 OK

            HttpEntity entity = response.getEntity();
            if (entity != null) {
                // return it as a String
                String result = EntityUtils.toString(entity);
                log.info(result);
            }

        } catch (ClientProtocolException e) {
            log.error("Clientprotocol exception: ", e);
        } catch (IOException e) {
            log.error("IOException: ", e);
        }
    }

    @Override public void afterTaskFailedEvent(TaskEvent taskEvent) {
        log.info("afterTaskFailedEvent {}", taskEvent);
    }

    @Override public void afterTaskAddedEvent(TaskEvent taskEvent) {
        log.info("afterTaskAddedEvent {}", taskEvent);

        HttpPost request = new HttpPost(apiUrl);

        // add request headers
        request.addHeader(HttpHeaders.ACCEPT, "application/json");
        request.addHeader(HttpHeaders.CONTENT_TYPE, "application/json");

        CreateTaskRequest taskRequest = new CreateTaskRequest();
        taskRequest.setSource(serverId);
        taskRequest.setComment("auto generated task from user task listener");
        taskRequest.setDirectTask(Boolean.FALSE);
        taskRequest.setPriority(Priority.MINOR);
        taskRequest.setTaskName(taskEvent.getTask().getName());
        Task task = taskEvent.getTask();
        if (task == null) {
            log.error("Task is empty");
            return;
        }
        InternalPeopleAssignments ipa = null;
        if (task.getPeopleAssignments() != null) {
            ipa = (InternalPeopleAssignments) task.getPeopleAssignments();
        }
        if (ipa != null && ipa.getRecipients() != null && ipa.getRecipients().size() > 0) {
            // fill recipients in the user reference
            UserReferenceDTO userReference = new UserReferenceDTO();
            userReference.setUserId(((InternalPeopleAssignments) taskEvent.getTask().getPeopleAssignments()).getRecipients().stream().map(OrganizationalEntity::getId).collect(Collectors.joining(",")));
            taskRequest.setUserReference(userReference);
        }
        if (ipa != null && ipa.getPotentialOwners() != null && ipa.getRecipients().size() > 0) {
            // fill potential owner in organization reference
            OrgaReferenceDTO orgaReference = new OrgaReferenceDTO();
            orgaReference.setOrgaId(taskEvent.getTask().getPeopleAssignments().getPotentialOwners().stream().map(OrganizationalEntity::getId).collect(Collectors.joining(",")));
            taskRequest.setOrgaReference(orgaReference);
        }

        if (task.getTaskData() == null) {
            log.warn("task data empty");
            return;
        }
        TaskData taskData = task.getTaskData();
        String callbackUrl = getCallbackUrl(taskData.getDeploymentId(), task.getId());
        taskRequest.setCallbackURL(callbackUrl);

        JBPMTaskDTO jbpmTask = new JBPMTaskDTO();
        jbpmTask.setDeploymentId(taskData.getDeploymentId());
        jbpmTask.setProcessId(taskData.getProcessId());
        jbpmTask.setProcessInstanceId(taskData.getProcessInstanceId());
        jbpmTask.setTaskInstanceId(task.getId());
        taskRequest.setJbpmTask(jbpmTask);

        //taskEvent.getTaskContext().loadTaskVariables(taskEvent.getTask());
        List<AdditionContentDTO> contentList = taskData.getTaskInputVariables().entrySet().stream().map(e -> new AdditionContentDTO(e.getKey(), e.getValue() + "")).collect(Collectors.toList());
        taskRequest.setAdditionalContentList(contentList);

        ObjectMapper mapper = new ObjectMapper();
        try {
            request.setEntity(new StringEntity(mapper.writeValueAsString(taskRequest)));
        } catch (UnsupportedEncodingException e) {
            log.error("Error creating json request", e);
            return;
        } catch (JsonProcessingException e) {
            log.error("Error creating json request", e);
            return;
        }

        try (
            CloseableHttpClient httpClient = HttpClients.createDefault();
            CloseableHttpResponse response = httpClient.execute(request)) {

            // Get HttpResponse Status
            log.info(response.getStatusLine().toString());        // HTTP/1.1 200 OK

            HttpEntity entity = response.getEntity();
            if (entity != null) {
                // return it as a String
                String result = EntityUtils.toString(entity);
                log.info(result);
            }

        } catch (ClientProtocolException e) {
            log.error("Clientprotocol exception: ", e);
        } catch (IOException e) {
            log.error("IOException: ", e);
        }
    }

    @Override public void afterTaskExitedEvent(TaskEvent taskEvent) {
        log.info("afterTaskExitedEvent {}", taskEvent);
    }

    @Override public void afterTaskReleasedEvent(TaskEvent taskEvent) {
        log.info("afterTaskReleasedEvent {}", taskEvent);
    }

    @Override public void afterTaskResumedEvent(TaskEvent taskEvent) {
        log.info("afterTaskResumedEvent {}", taskEvent);
    }

    @Override public void afterTaskSuspendedEvent(TaskEvent taskEvent) {
        log.info("afterTaskSuspendedEvent {}", taskEvent);
    }

    @Override public void afterTaskForwardedEvent(TaskEvent taskEvent) {
        log.info("afterTaskForwardedEvent {}", taskEvent);
    }

    @Override public void afterTaskDelegatedEvent(TaskEvent taskEvent) {
        log.info("afterTaskDelegatedEvent {}", taskEvent);
    }

    @Override public void afterTaskNominatedEvent(TaskEvent taskEvent) {
        log.info("afterTaskNominatedEvent {}", taskEvent);
    }

    private String getCallbackUrl(String deploymentId, Long taskId) {
        if (kieApiUrl == null) {
            StringBuilder sb = new StringBuilder();
            log.info("Callback kie server url null. Creation of callback url is dynamic.");
            if (namespace != null) {
                sb.append("http://");
                sb.append(serverId).append(".");
                sb.append(namespace).append(".svc.cluster.local:8080");
                log.info("Namespace was defined using local openshift url for callback.");
            } else {
                String url = System.getenv("KIE_SERVER_LOCATION");
                if (url != null) {
                    log.info("Kie server location found {}. Set as callback url", url);
                    sb.append(url);
                } else {
                    sb.append("http://");
                    sb.append(serverId).append(":8080/");
                    log.info("Kie server location not defined using local url");
                }
            }
            kieApiUrl = sb.toString();
            log.info("Callback url set to {}", kieApiUrl);
        }

        StringBuilder sb = new StringBuilder(kieApiUrl);
        sb.append("/services/rest/server/containers/").append(deploymentId);
        sb.append("/tasks/").append(taskId).append("/states");

        return sb.toString();
    }
}
