/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.tomitribe.churchkey;

import org.tomitribe.churchkey.dsa.Dsa;
import org.tomitribe.churchkey.jwk.JwkParser;
import org.tomitribe.churchkey.pem.PemParser;
import org.tomitribe.churchkey.rsa.Rsa;
import org.tomitribe.churchkey.ssh.OpenSSHParser;
import org.tomitribe.churchkey.ssh.SSH2Parser;

import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

public class Key {

    private final java.security.Key key;
    private final Type type;
    private final Algorithm algorithm;
    private final Format format;
    private final Map<String, String> attributes = new HashMap<>();
    private final Key publicKey;

    public Key(final java.security.Key key, final java.security.PublicKey publicKey, final Type type, final Algorithm algorithm, final Format format) {
        this(key, publicKey, type, algorithm, format, new HashMap<>());
    }

    public Key(final java.security.Key key, final Type type, final Algorithm algorithm, final Format format) {
        this(key, type, algorithm, format, new HashMap<>());
    }

    public Key(final java.security.Key key, final Type type, final Algorithm algorithm, final Format format, final Map<String, String> attributes) {
        this(key, null, type, algorithm, format, attributes);
    }

    public Key(final java.security.Key key, final java.security.PublicKey publicKey, final Type type,
               final Algorithm algorithm, final Format format, final Map<String, String> attributes) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(type);
        Objects.requireNonNull(algorithm);
        Objects.requireNonNull(format);
        Objects.requireNonNull(attributes);
        this.key = key;
        this.type = type;
        this.algorithm = algorithm;
        this.format = format;
        this.attributes.putAll(attributes);
        this.publicKey = publicKey(key, publicKey, algorithm, format, attributes);
    }

    private Key publicKey(final java.security.Key key, PublicKey publicKey, final Algorithm algorithm,
                          final Format format, final Map<String, String> attributes) {
        if (!(key instanceof PrivateKey)) {
            return null;
        }

        if (publicKey == null) {
            // For RSA and DSA it's fairly easy to calculate it
            if (algorithm == Algorithm.RSA && key instanceof RSAPrivateCrtKey) {
                final RSAPrivateCrtKey rsaPrivateCrtKey = (RSAPrivateCrtKey) key;
                publicKey = Rsa.Public.builder()
                        .publicExponent(rsaPrivateCrtKey.getPublicExponent())
                        .modulus(rsaPrivateCrtKey.getModulus())
                        .build().toKey();
            } else if (algorithm == Algorithm.DSA && key instanceof DSAPrivateKey) {
                final DSAPrivateKey privateKey = (DSAPrivateKey) key;
                publicKey = Dsa.Private.builder()
                        .x(privateKey.getX())
                        .p(privateKey.getParams().getP())
                        .g(privateKey.getParams().getG())
                        .q(privateKey.getParams().getQ())
                        .build()
                        .toPublic()
                        .toKey();
            }
        }

        if (publicKey != null) {
            return new Key(publicKey, Type.PUBLIC, algorithm, format, attributes);
        }

        return null;
    }

    public Key getPublicKey() {
        if (type == Type.PRIVATE) return publicKey;
        throw new IllegalStateException(type + " keys do not have public keys");
    }

    public Map<String, String> getAttributes() {
        return attributes;
    }

    public String getAttribute(final String name) {
        return attributes.get(name);
    }

    public boolean hasAttribute(final String name) {
        return attributes.containsKey(name);
    }

    public java.security.Key getKey() {
        return key;
    }

    public Type getType() {
        return type;
    }

    public Algorithm getAlgorithm() {
        return algorithm;
    }

    public Format getFormat() {
        return format;
    }

    // TODO
    // public Key getPublicKey()

    public byte[] encode(final Format format) {
        return format.encode(this);
    }

    public enum Type {
        PUBLIC,
        PRIVATE,
        SECRET
    }

    public enum Algorithm {
        RSA, DSA, EC, OCT;

        public Factory getKeyFactory() {
            if (this == OCT) throw new UnsupportedOperationException();
            try {
                return new Factory(KeyFactory.getInstance(name()));
            } catch (NoSuchAlgorithmException e) {
                throw new UnsupportedAlgorithmException(this, e);
            }
        }
    }

    public enum Format {
        JWK(new JwkParser()),
        OPENSSH(new OpenSSHParser()),
        SSH2(new SSH2Parser()),
        PEM(new PemParser()),
        ;

        private final Parser parser;

        Format(final Parser parser) {
            this.parser = parser;
        }

        public byte[] encode(final Key key) {
            return parser.encode(key);
        }

        public Key decode(final byte[] bytes) {
            return parser.decode(bytes);
        }

        public interface Parser {
            Key decode(final byte[] bytes);

            byte[] encode(final Key key);
        }
    }

    public static class Factory {
        private final KeyFactory keyFactory;

        public Factory(final KeyFactory keyFactory) {
            this.keyFactory = keyFactory;
        }

        public PublicKey generatePublic(final KeySpec keySpec) {
            try {
                return keyFactory.generatePublic(keySpec);
            } catch (InvalidKeySpecException e) {
                throw new InvalidPublicKeySpecException(keySpec, e);
            }
        }

        public PrivateKey generatePrivate(final KeySpec keySpec) {
            try {
                return keyFactory.generatePrivate(keySpec);
            } catch (InvalidKeySpecException e) {
                throw new InvalidPrivateKeySpecException(keySpec, e);
            }
        }
    }

}
