package org.tresql

import scala.language.higherKinds

trait Typed { this: RowLike =>
  def typedPf(columnIndex: Int): PartialFunction[String, Any] = {
    case "Int" | "int"                => int(columnIndex)
    case "Long" | "long"              => long(columnIndex)
    case "Double" | "double"          => double(columnIndex)
    case "Boolean" | "boolean"        => boolean(columnIndex)
    case "scala.math.BigDecimal"      => bigdecimal(columnIndex)
    case "scala.math.BigInt"          => bigint(columnIndex)
    case "java.lang.String"           => string(columnIndex)
    case "java.util.Date"             => timestamp(columnIndex)
    case "java.sql.Date"              => date(columnIndex)
    case "java.sql.Timestamp"         => timestamp(columnIndex)
    case "java.sql.Time"              => time(columnIndex)
    case "java.time.LocalDate"        => Option(date(columnIndex)).map(_.toLocalDate).orNull
    case "java.time.LocalDateTime"    => Option(timestamp(columnIndex)).map(_.toLocalDateTime).orNull
    case "java.time.LocalTime"        => Option(time(columnIndex)).map(_.toLocalTime).orNull
    case "java.lang.Integer"          => jInt(columnIndex)
    case "java.lang.Long"             => jLong(columnIndex)
    case "java.lang.Double"           => jDouble(columnIndex)
    case "java.math.BigDecimal"       => jBigDecimal(columnIndex)
    case "java.math.BigInteger"       => jBigInteger(columnIndex)
    case "java.lang.Boolean"          => jBoolean(columnIndex)
    case "Array[Byte]"                => bytes(columnIndex)
    case "java.io.InputStream"        => stream(columnIndex)
    case "java.io.Reader"             => reader(columnIndex)
    case "java.sql.Blob"              => blob(columnIndex)
    case "java.sql.Clob"              => clob(columnIndex)
    case "java.sql.Array"             => array(columnIndex)
  }
  def typed(columnIndex: Int, manifestName: String): Any =
    (typedPf(columnIndex) orElse ({ case _ => apply(columnIndex) }: PartialFunction[String, Any]))(manifestName)

  def typed[T](columnIndex: Int)(implicit m: Manifest[T]): T = m.toString match {
    case x if x.startsWith("scala.Tuple") =>
      typed((CoreTypes.convTuple[Product] _).asInstanceOf[CoreTypes.Converter[T]], m)
    case x if x.startsWith("scala.collection.immutable.List") && m.typeArguments.size == 1 =>
      result(columnIndex).map(_.typed(0)(m.typeArguments.head)).toList.asInstanceOf[T]
    case x => typed(columnIndex, x).asInstanceOf[T]
  }

  def typed[T:Manifest](name: String): T

  def typed[T](implicit converter: CoreTypes.Converter[T], m: Manifest[T]): T =
    if (converter != null) converter(this, m) else typed(0).asInstanceOf[T]
}

trait TypedResult[+R <: RowLike] { this: Result[R] =>

  def head[T: Manifest](implicit converter: CoreTypes.Converter[T]): T =
    headOption[T].getOrElse(throw new NoSuchElementException("No rows in result"))

  def headOption[T: Manifest](implicit converter: CoreTypes.Converter[T]): Option[T] =
    try
      if (hasNext) {
        next()
        Some(this.asInstanceOf[RowLike].typed[T])
      } else None
    finally close

  /** Used by tresql scala macro to convert result to single value */
  def headValue(typeName: String): Any =
    headOptionValue(typeName).getOrElse(throw new NoSuchElementException("No rows in result"))
  /** Used by tresql scala macro to convert result to single value */
  def headOptionValue(typeName: String): Option[Any] =
    try
      if (hasNext) {
        next()
        Some(this.asInstanceOf[RowLike].typed(0, typeName))
      } else None
    finally close

  def unique[T: Manifest](implicit converter: CoreTypes.Converter[T]): T =
    uniqueOption[T].getOrElse(throw new NoSuchElementException("No rows in result"))

  def uniqueOption[T: Manifest](implicit converter: CoreTypes.Converter[T]): Option[T] =
    try if (hasNext) {
      next()
      val v = this.asInstanceOf[RowLike].typed[T]
      if (hasNext) throw new TooManyRowsException("More than one row for unique result")
      else Some(v)
    } else None
    finally close

  def list[T: Manifest](implicit converter: CoreTypes.Converter[T]): List[T] =
    this.map(r => this.asInstanceOf[RowLike].typed[T]).toList

  import CoreTypes._
  //--------------- GENERATED CODE------------------//
  def head[T1, T2](implicit m: Manifest[(T1, T2)]): (T1, T2) = head[(T1, T2)]
  def head[T1, T2, T3](implicit m: Manifest[(T1, T2, T3)]): (T1, T2, T3) = head[(T1, T2, T3)]
  def head[T1, T2, T3, T4](implicit m: Manifest[(T1, T2, T3, T4)]): (T1, T2, T3, T4) = head[(T1, T2, T3, T4)]
  def head[T1, T2, T3, T4, T5](implicit m: Manifest[(T1, T2, T3, T4, T5)]): (T1, T2, T3, T4, T5) = head[(T1, T2, T3, T4, T5)]
  def head[T1, T2, T3, T4, T5, T6](implicit m: Manifest[(T1, T2, T3, T4, T5, T6)]): (T1, T2, T3, T4, T5, T6) = head[(T1, T2, T3, T4, T5, T6)]
  def head[T1, T2, T3, T4, T5, T6, T7](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7)]): (T1, T2, T3, T4, T5, T6, T7) = head[(T1, T2, T3, T4, T5, T6, T7)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)]): (T1, T2, T3, T4, T5, T6, T7, T8) = head[(T1, T2, T3, T4, T5, T6, T7, T8)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]
  def headOption[T1, T2](implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)]): Option[(T1, T2)] = headOption[(T1, T2)]
  def headOption[T1, T2, T3](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)]): Option[(T1, T2, T3)] = headOption[(T1, T2, T3)]
  def headOption[T1, T2, T3, T4](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)]): Option[(T1, T2, T3, T4)] = headOption[(T1, T2, T3, T4)]
  def headOption[T1, T2, T3, T4, T5](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)]): Option[(T1, T2, T3, T4, T5)] = headOption[(T1, T2, T3, T4, T5)]
  def headOption[T1, T2, T3, T4, T5, T6](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)]): Option[(T1, T2, T3, T4, T5, T6)] = headOption[(T1, T2, T3, T4, T5, T6)]
  def headOption[T1, T2, T3, T4, T5, T6, T7](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)]): Option[(T1, T2, T3, T4, T5, T6, T7)] = headOption[(T1, T2, T3, T4, T5, T6, T7)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]
  def unique[T1, T2](implicit m: Manifest[(T1, T2)]): (T1, T2) = unique[(T1, T2)]
  def unique[T1, T2, T3](implicit m: Manifest[(T1, T2, T3)]): (T1, T2, T3) = unique[(T1, T2, T3)]
  def unique[T1, T2, T3, T4](implicit m: Manifest[(T1, T2, T3, T4)]): (T1, T2, T3, T4) = unique[(T1, T2, T3, T4)]
  def unique[T1, T2, T3, T4, T5](implicit m: Manifest[(T1, T2, T3, T4, T5)]): (T1, T2, T3, T4, T5) = unique[(T1, T2, T3, T4, T5)]
  def unique[T1, T2, T3, T4, T5, T6](implicit m: Manifest[(T1, T2, T3, T4, T5, T6)]): (T1, T2, T3, T4, T5, T6) = unique[(T1, T2, T3, T4, T5, T6)]
  def unique[T1, T2, T3, T4, T5, T6, T7](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7)]): (T1, T2, T3, T4, T5, T6, T7) = unique[(T1, T2, T3, T4, T5, T6, T7)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)]): (T1, T2, T3, T4, T5, T6, T7, T8) = unique[(T1, T2, T3, T4, T5, T6, T7, T8)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](implicit m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]
  def uniqueOption[T1, T2](implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)]): Option[(T1, T2)] = uniqueOption[(T1, T2)]
  def uniqueOption[T1, T2, T3](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)]): Option[(T1, T2, T3)] = uniqueOption[(T1, T2, T3)]
  def uniqueOption[T1, T2, T3, T4](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)]): Option[(T1, T2, T3, T4)] = uniqueOption[(T1, T2, T3, T4)]
  def uniqueOption[T1, T2, T3, T4, T5](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)]): Option[(T1, T2, T3, T4, T5)] = uniqueOption[(T1, T2, T3, T4, T5)]
  def uniqueOption[T1, T2, T3, T4, T5, T6](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)]): Option[(T1, T2, T3, T4, T5, T6)] = uniqueOption[(T1, T2, T3, T4, T5, T6)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)]): Option[(T1, T2, T3, T4, T5, T6, T7)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]
  def list[T1, T2](implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)]): List[(T1, T2)] = list[(T1, T2)]
  def list[T1, T2, T3](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)]): List[(T1, T2, T3)] = list[(T1, T2, T3)]
  def list[T1, T2, T3, T4](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)]): List[(T1, T2, T3, T4)] = list[(T1, T2, T3, T4)]
  def list[T1, T2, T3, T4, T5](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)]): List[(T1, T2, T3, T4, T5)] = list[(T1, T2, T3, T4, T5)]
  def list[T1, T2, T3, T4, T5, T6](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)]): List[(T1, T2, T3, T4, T5, T6)] = list[(T1, T2, T3, T4, T5, T6)]
  def list[T1, T2, T3, T4, T5, T6, T7](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)]): List[(T1, T2, T3, T4, T5, T6, T7)] = list[(T1, T2, T3, T4, T5, T6, T7)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)]): List[(T1, T2, T3, T4, T5, T6, T7, T8)] = list[(T1, T2, T3, T4, T5, T6, T7, T8)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)]
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)]
  //--------------- END OF GENERATED CODE------------------//
}

trait TypedQuery {this: Query =>
  def head[T: Manifest](expr: String, params: Any*)(implicit converter: CoreTypes.Converter[T], r: Resources): T = {
    apply(expr, params: _*).head[T]
  }
  def headOption[T: Manifest](expr: String, params: Any*)(implicit converter: CoreTypes.Converter[T],
                                                          r: Resources): Option[T] = {
    apply(expr, params: _*).headOption[T]
  }
  def unique[T: Manifest](expr: String, params: Any*)(implicit converter: CoreTypes.Converter[T],
                                                      r: Resources): T = {
    apply(expr, params: _*).unique[T]
  }
  def uniqueOption[T: Manifest](expr: String, params: Any*)(implicit converter: CoreTypes.Converter[T],
                                                            r: Resources): Option[T] = {
    apply(expr, params: _*).uniqueOption[T]
  }

  def toListOfMaps(expr: String, params: Any*)(implicit r: Resources): List[Map[String, Any]] =
    apply(expr, params: _*).toListOfMaps

  def list[T: Manifest](expr: String, params: Any*)(implicit converter: CoreTypes.Converter[T], r: Resources): List[T] =
    apply(expr, params: _*).list[T]

  import CoreTypes._
  //--------------- GENERATED CODE------------------//
  def head[T1, T2](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)], r: Resources): (T1, T2) = head[(T1, T2)](expr, params: _*)
  def head[T1, T2, T3](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)], r: Resources): (T1, T2, T3) = head[(T1, T2, T3)](expr, params: _*)
  def head[T1, T2, T3, T4](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)], r: Resources): (T1, T2, T3, T4) = head[(T1, T2, T3, T4)](expr, params: _*)
  def head[T1, T2, T3, T4, T5](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)], r: Resources): (T1, T2, T3, T4, T5) = head[(T1, T2, T3, T4, T5)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)], r: Resources): (T1, T2, T3, T4, T5, T6) = head[(T1, T2, T3, T4, T5, T6)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)], r: Resources): (T1, T2, T3, T4, T5, T6, T7) = head[(T1, T2, T3, T4, T5, T6, T7)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8) = head[(T1, T2, T3, T4, T5, T6, T7, T8)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)](expr, params: _*)
  def head[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22) = head[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)](expr, params: _*)
  def headOption[T1, T2](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)], r: Resources): Option[(T1, T2)] = headOption[(T1, T2)](expr, params: _*)
  def headOption[T1, T2, T3](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)], r: Resources): Option[(T1, T2, T3)] = headOption[(T1, T2, T3)](expr, params: _*)
  def headOption[T1, T2, T3, T4](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)], r: Resources): Option[(T1, T2, T3, T4)] = headOption[(T1, T2, T3, T4)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)], r: Resources): Option[(T1, T2, T3, T4, T5)] = headOption[(T1, T2, T3, T4, T5)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)], r: Resources): Option[(T1, T2, T3, T4, T5, T6)] = headOption[(T1, T2, T3, T4, T5, T6)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7)] = headOption[(T1, T2, T3, T4, T5, T6, T7)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)](expr, params: _*)
  def headOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)] = headOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)](expr, params: _*)
  def unique[T1, T2](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)], r: Resources): (T1, T2) = unique[(T1, T2)](expr, params: _*)
  def unique[T1, T2, T3](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)], r: Resources): (T1, T2, T3) = unique[(T1, T2, T3)](expr, params: _*)
  def unique[T1, T2, T3, T4](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)], r: Resources): (T1, T2, T3, T4) = unique[(T1, T2, T3, T4)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)], r: Resources): (T1, T2, T3, T4, T5) = unique[(T1, T2, T3, T4, T5)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)], r: Resources): (T1, T2, T3, T4, T5, T6) = unique[(T1, T2, T3, T4, T5, T6)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)], r: Resources): (T1, T2, T3, T4, T5, T6, T7) = unique[(T1, T2, T3, T4, T5, T6, T7)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8) = unique[(T1, T2, T3, T4, T5, T6, T7, T8)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)](expr, params: _*)
  def unique[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)], r: Resources): (T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22) = unique[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)](expr, params: _*)
  def uniqueOption[T1, T2](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)], r: Resources): Option[(T1, T2)] = uniqueOption[(T1, T2)](expr, params: _*)
  def uniqueOption[T1, T2, T3](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)], r: Resources): Option[(T1, T2, T3)] = uniqueOption[(T1, T2, T3)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)], r: Resources): Option[(T1, T2, T3, T4)] = uniqueOption[(T1, T2, T3, T4)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)], r: Resources): Option[(T1, T2, T3, T4, T5)] = uniqueOption[(T1, T2, T3, T4, T5)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)], r: Resources): Option[(T1, T2, T3, T4, T5, T6)] = uniqueOption[(T1, T2, T3, T4, T5, T6)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)](expr, params: _*)
  def uniqueOption[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)], r: Resources): Option[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)] = uniqueOption[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)](expr, params: _*)
  def list[T1, T2](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m: Manifest[(T1, T2)], r: Resources): List[(T1, T2)] = list[(T1, T2)](expr, params: _*)
  def list[T1, T2, T3](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m: Manifest[(T1, T2, T3)], r: Resources): List[(T1, T2, T3)] = list[(T1, T2, T3)](expr, params: _*)
  def list[T1, T2, T3, T4](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m: Manifest[(T1, T2, T3, T4)], r: Resources): List[(T1, T2, T3, T4)] = list[(T1, T2, T3, T4)](expr, params: _*)
  def list[T1, T2, T3, T4, T5](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m: Manifest[(T1, T2, T3, T4, T5)], r: Resources): List[(T1, T2, T3, T4, T5)] = list[(T1, T2, T3, T4, T5)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m: Manifest[(T1, T2, T3, T4, T5, T6)], r: Resources): List[(T1, T2, T3, T4, T5, T6)] = list[(T1, T2, T3, T4, T5, T6)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m: Manifest[(T1, T2, T3, T4, T5, T6, T7)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7)] = list[(T1, T2, T3, T4, T5, T6, T7)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8)] = list[(T1, T2, T3, T4, T5, T6, T7, T8)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21)](expr, params: _*)
  def list[T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22](expr: String, params: Any*)(implicit m1: Manifest[T1], m2: Manifest[T2], m3: Manifest[T3], m4: Manifest[T4], m5: Manifest[T5], m6: Manifest[T6], m7: Manifest[T7], m8: Manifest[T8], m9: Manifest[T9], m10: Manifest[T10], m11: Manifest[T11], m12: Manifest[T12], m13: Manifest[T13], m14: Manifest[T14], m15: Manifest[T15], m16: Manifest[T16], m17: Manifest[T17], m18: Manifest[T18], m19: Manifest[T19], m20: Manifest[T20], m21: Manifest[T21], m22: Manifest[T22], m: Manifest[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)], r: Resources): List[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)] = list[(T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, T13, T14, T15, T16, T17, T18, T19, T20, T21, T22)](expr, params: _*)
  //--------------- END OF GENERATED CODE------------------//
}
