package org.tuzhao.library.debug;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;

import org.tuzhao.library.tools.FormatUtil;
import org.tuzhao.library.tools.TimeZoneUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

/**
 * 处理异常基类
 * @author tuzhao
 *         Created by tuzhao on 16-3-25.
 */
public abstract class BaseCrashHandler<T> implements Thread.UncaughtExceptionHandler {

	private final Context context;
	private final HashMap<String, String> infoMap = new HashMap<>();
	private final Thread.UncaughtExceptionHandler mDefaultHandler;
	private final T t;

	public BaseCrashHandler(Context context, T t) {
		this.context = context;
		this.t = t;
		mDefaultHandler = Thread.getDefaultUncaughtExceptionHandler();
		Thread.setDefaultUncaughtExceptionHandler(this);
	}

	@Override
	public void uncaughtException(Thread thread, Throwable ex) {
		beforeExceptionDeal(thread, ex);
		if (!handleException(ex) & mDefaultHandler != null) {
			it("uncaughtException", "自定义异常处理失败");
			caughtExceptionFailure();
			mDefaultHandler.uncaughtException(thread, ex);
		} else {
			it("uncaughtException", "自定义异常处理成功");
			caughtExceptionSucc();
		}
	}

	//自定义异常处理方法
	private boolean handleException(Throwable ex) {
		if (ex == null) {
			return false;
		}
		String threadName = "threadName=" + Thread.currentThread().getName();
		String threadGroup = "threadGroup=" + Thread.currentThread().getThreadGroup().toString();
		String devInfo = threadName + "\n" + threadGroup + "\n" + getDeviceInfo(context);
		String crashInfo = getCrashInfo(ex);

		it("handleException", "设备信息:\n" + devInfo);
		it("handleException", "异常信息:" + crashInfo);

		saveCrashInfo(devInfo, crashInfo);
		API_UploadErrorInfo(devInfo, crashInfo);
		return true;
	}

	private void saveCrashInfo(String devInfo, String crashInfo) {
		if (devInfo == null | crashInfo == null) {
			it("saveCrashInfo", "save crashInfo failure!");
			return;
		}
		String path;
		if (Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
			//noinspection ConstantConditions
			path = context.getExternalCacheDir().getAbsolutePath() + "/" + "exceptionLog";
		} else {
			path = context.getCacheDir().getAbsolutePath() + "/" + "exceptionLog";
		}

		File file = new File(path);
		if (!file.exists()) {
			//noinspection ResultOfMethodCallIgnored
			file.mkdirs();
		}

		String fileName = FormatUtil.getFileName(".txt");

		FileOutputStream out = null;
		try {
			out = new FileOutputStream(path + "/" + fileName);
			out.write((devInfo + crashInfo).getBytes());
			it("异常信息保存成功", path + fileName);
		} catch (Exception e) {
			ie(e);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (IOException e) {
					ie(e);
				}
			}
		}
	}

	//获得设备信息
	private String getDeviceInfo(Context context) {
		StringBuilder builder = new StringBuilder();

		int sdk = Build.VERSION.SDK_INT; // SDK号
		String model = Build.MODEL; // 手机型号
		String release = Build.VERSION.RELEASE; // android系统版本号

		builder.append("sdk号:").append(sdk).append(";\n");
		builder.append("手机型号:").append(model).append(";\n");
		builder.append("android系统版本号:").append(release).append(";\n");

		PackageManager packageManager = context.getPackageManager();
		try {
			PackageInfo packageInfo = packageManager.getPackageInfo(context.getPackageName(), PackageManager.GET_ACTIVITIES);
			if (packageInfo != null) {
				String versionName = packageInfo.versionName == null ? "versionName为null" : packageInfo.versionName;
				String versioncode = packageInfo.versionCode + "";
				infoMap.put("versionName", versionName);
				infoMap.put("versionCode", versioncode);
			}
			Field[] fields = Build.class.getDeclaredFields();
			for (Field field : fields) {
				field.setAccessible(true);
				infoMap.put(field.getName(), field.get(null).toString());
			}
		} catch (Exception e) {
			ie(e);
		}

		//这种形式获得键值对
		for (Map.Entry<String, String> entry : infoMap.entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();
			builder.append(key).append("=").append(value).append("\n");
		}
		return builder.toString();
	}

	//获得异常信息
	private String getCrashInfo(Throwable ex) {
		StringBuilder builder = new StringBuilder();

		builder.append(FormatUtil.formatTime1(String.valueOf(TimeZoneUtil.getChinaCurrentMils())));
		builder.append("\n");

		Writer writer = new StringWriter();
		PrintWriter printWriter = new PrintWriter(writer);

		ex.printStackTrace(printWriter);
		Throwable cause = ex.getCause();
		while (cause != null) {
			cause.printStackTrace(printWriter);
			cause = cause.getCause();
		}
		printWriter.close();
		String result = writer.toString();
		builder.append(result);
		return builder.toString();
	}

	public Context getContext() {
		return context;
	}

	public T getT() {
		return t;
	}

	public abstract void beforeExceptionDeal(Thread thread, Throwable ex);

	/**
	 * 自定义上传异常到服务器的方法
	 */
	public abstract void API_UploadErrorInfo(String devInfo, String crashInfo);

	/**
	 * 异常处理成功的方法
	 */
	public abstract void caughtExceptionSucc();

	/**
	 * 异常处理失败的方法
	 */
	public abstract void caughtExceptionFailure();

	public void it(String tag, Object msg) {
		LogManager.getDefaultLogger().it(tag, msg);
	}

	public void ie(Exception e) {
		LogManager.getDefaultLogger().e(BaseCrashHandler.class.getSimpleName(), e);
	}

}
