package org.tuzhao.library.tools;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Bitmap.CompressFormat;
import android.graphics.BitmapFactory;
import android.graphics.BitmapFactory.Options;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.PixelFormat;
import android.graphics.Rect;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.media.ExifInterface;
import android.os.Environment;
import android.util.Base64;
import android.util.Log;

import org.tuzhao.library.debug.LogManager;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;


/**
 * 图像相关处理工具类
 * @author tuzhao
 */
public class BitmapUtil {

	/**
	 * 获得图像的Options
	 * @param path path
	 * @return options
	 */
	public static Options getOptions(String path) {
		Options options = new Options();
		options.inJustDecodeBounds = true;
		BitmapFactory.decodeFile(path, options);
		return options;
	}


	/**
	 * 获得指定图片路径的Bitmap对象
	 * @param strImgPath path
	 * @return Bitmap
	 */
	public static Bitmap getBitmapByPath(String strImgPath) {
		Bitmap localBitmap = null;
		Options localOptions = new Options();
		try {
			localOptions.inJustDecodeBounds = true;
			BitmapFactory.decodeFile(strImgPath, localOptions);
			localOptions.inJustDecodeBounds = false;
			int i = (int) (localOptions.outHeight / 800.0F);
			if (i <= 0) i = 2;
			localOptions.inSampleSize = i;
			localBitmap = BitmapFactory.decodeFile(strImgPath, localOptions);
			return localBitmap;
		} catch (Exception e) {
			Log.e("changchun", "图片解析错误" + e.toString());
			e.printStackTrace();
			localOptions.inSampleSize = 5;
			localBitmap = BitmapFactory.decodeFile(strImgPath, localOptions);
		}
		return localBitmap;
	}


	/**
	 * 通过指定图片路径获得Bitmap对象
	 * @param path         path
	 * @param screenWidth  width
	 * @param screenHeight height
	 * @return Bitmap
	 * @throws FileNotFoundException not found file
	 */
	public static Bitmap getBitmapByPath(String path, int screenWidth, int screenHeight) throws FileNotFoundException {
		File file = new File(path);

		if (!file.exists()) {
			throw new FileNotFoundException();
		}

		FileInputStream in = null;
		Bitmap bitmap = null;
		try {
			in = new FileInputStream(file);

			Options options = getOptions(path);

			if (options != null) {
				int maxSize = screenWidth > screenHeight ? screenWidth : screenHeight;
				int inSimpleSize = computeSampleSize(options, maxSize, screenWidth * screenHeight);

				options.inSampleSize = inSimpleSize; // 设置缩放比例
				options.inJustDecodeBounds = false;
			}

			bitmap = BitmapFactory.decodeStream(in, null, options);

		} catch (IOException e) {

		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

		return bitmap;

	}


	private static int computeSampleSize(Options options, int minSideLength, int maxNumOfPixels) {
		int initialSize = computeInitialSampleSize(options, minSideLength, maxNumOfPixels);
		int roundedSize;
		if (initialSize <= 8) {
			roundedSize = 1;
			while (roundedSize < initialSize) {
				roundedSize <<= 1;
			}
		} else {
			roundedSize = (initialSize + 7) / 8 * 8;
		}

		return roundedSize;
	}


	private static int computeInitialSampleSize(Options options, int minSideLength, int maxNumOfPixels) {
		double w = options.outWidth;
		double h = options.outHeight;

		int lowerBound = (maxNumOfPixels == -1) ? 1 : (int) Math.ceil(Math.sqrt(w * h / maxNumOfPixels));
		int upperBound = (minSideLength == -1) ? 128 : (int) Math.min(Math.floor(w / minSideLength), Math.floor(h / minSideLength));

		if (upperBound < lowerBound) {
			// return the larger one when there is no overlapping zone.
			return lowerBound;
		}

		if ((maxNumOfPixels == -1) && (minSideLength == -1)) {
			return 1;
		} else if (minSideLength == -1) {
			return lowerBound;
		} else {
			return upperBound;
		}
	}


	/**
	 * 将byte数组转换为bitmap
	 * @param b byte
	 * @return bitmap
	 */
	public static Bitmap getBitmapFromByte(byte[] b) {
		if (b == null) {
			return null;
		}
		Bitmap bitmap = BitmapFactory.decodeByteArray(b, 0, b.length);
		return bitmap;
	}


	/**
	 * 将指定路径的图片转换为Base64字符串
	 * @param picPath picPath
	 * @return String
	 */
	public static String getBase64StringByPath(String picPath) {
		InputStream in = null;
		byte[] data = null;
		try {
			in = new FileInputStream(picPath);
			data = new byte[in.available()];
			in.read(data);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

		String picString = Base64.encodeToString(data, Base64.DEFAULT);

		return picString;

	}


	/**
	 * 将指定路径的照片转换为Base64编码的字符串
	 * @param picPath      照片路径
	 * @param qualityValue qualityValue
	 * @return picString 转换后的字符
	 */
	public static String getBase64StringByPath(String picPath, int qualityValue) {
		try {

			Options localOptions = new Options();
			localOptions.inJustDecodeBounds = true;
			BitmapFactory.decodeFile(picPath, localOptions);
			localOptions.inJustDecodeBounds = false;
			int i = (int) (localOptions.outHeight / 800.0F);
			if (i <= 0) {
				i = 4;
			}

			localOptions.inSampleSize = i;
			Bitmap bp = BitmapFactory.decodeFile(picPath, localOptions);
			ByteArrayOutputStream bStream = new ByteArrayOutputStream();
			if (null != bp) {
				bp.compress(CompressFormat.JPEG, qualityValue, bStream);
				byte[] picByte = bStream.toByteArray();
				String picString = Base64.encodeToString(picByte, Base64.DEFAULT);
				return picString;
			} else {
				return null;
			}

		} catch (Exception e) {
			Log.e("changchun", "图片转换失败--图片太大");
			e.printStackTrace();
		}
		return null;
	}


	/**
	 * 将bitmap转换为byte[]
	 * @param bitmap  bitmap
	 * @param quality quality
	 * @return byte[]
	 */
	public static byte[] getByte(Bitmap bitmap, int quality) {
		ByteArrayOutputStream out = null;
		try {
			out = new ByteArrayOutputStream();
			bitmap.compress(CompressFormat.JPEG, quality, out);
			out.flush();
			return out.toByteArray();
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

	}


	/**
	 * 将Base64字符串转换为图片存储到指定路径
	 * @param picString picString
	 * @param picPath   picPath
	 */
	public static void getPicFromBase64String(String picString, String picPath) {
		try {

			byte[] picByte = Base64.decode(picString, Base64.DEFAULT);
			Bitmap bp = BitmapFactory.decodeByteArray(picByte, 0, picByte.length);
			FileOutputStream fos = new FileOutputStream(picPath);
			bp.compress(CompressFormat.JPEG, 100, fos);
			fos.close();

		} catch (Exception e) {
			e.printStackTrace();
		}

	}


	/**
	 * 获得指定路径图片旋转的角度 若没有旋转则返回0
	 * @param path 指定的图片路径
	 * @return int degree  //返回当前图片被旋转的度数
	 */
	public static int getPicAngle(String path) {
		if (path == null) {
			return 0;
		}
		int degree = 0;
		try {
			ExifInterface exifInterface = new ExifInterface(path);
			int orientation = exifInterface.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
			switch (orientation) {
				case ExifInterface.ORIENTATION_ROTATE_90:
					degree = 90;
					break;
				case ExifInterface.ORIENTATION_ROTATE_180:
					degree = 180;
					break;
				case ExifInterface.ORIENTATION_ROTATE_270:
					degree = 270;
					break;
			}
		} catch (IOException e) {
			e.printStackTrace();
			return 0;
		}
		return degree;
	}


	/**
	 * 旋转图片(Bitmap对象)
	 * @param angle  旋转的角度
	 * @param bitmap 需要旋转的bitmap对象
	 * @return Bitmap  经过旋转后的bitmap
	 */
	public static Bitmap rotateBitmap(int angle, Bitmap bitmap) {
		// 旋转图片 动作
		Matrix matrix = new Matrix();
		matrix.postRotate(angle);
		// 创建新的图片
		Bitmap resizedBitmap = Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), matrix, true);
		it("rotateBitmap", "correct degree-->" + angle);
		return resizedBitmap;
	}


	/**
	 * 旋转图像(drawable中的图片)
	 * @param context context
	 * @param imgId   imgId
	 * @param angle   angle
	 * @return bitmap
	 */
	public static Bitmap rotateBitmap(Context context, int imgId, int angle) {

		Bitmap bitmap = BitmapFactory.decodeResource(context.getResources(), imgId);
		Matrix matrix = new Matrix();
		matrix.postRotate(angle);
		bitmap = Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), matrix, true);
		return bitmap;

	}


	/**
	 * 缩放图像(Bitamp对象)到指定大小
	 * @param bitmap bitmap
	 * @param width  width
	 * @param height height
	 * @return bitmap
	 */
	public static Bitmap zoomBitmap(Bitmap bitmap, int width, int height) {
		int localWidth = bitmap.getWidth();
		int localHeight = bitmap.getHeight();

		Matrix matrix = new Matrix();
		float scaleWidth = ((float) width) / localWidth;
		float scaleHeight = ((float) height) / localHeight;

		matrix.postScale(scaleWidth, scaleHeight);
		bitmap = Bitmap.createBitmap(bitmap, 0, 0, localWidth, localHeight, matrix, true);

		return bitmap;

	}


	/**
	 * 裁剪图像   这其实不是严格意义上的裁剪  只是将原图像的部分显示在界面上
	 * @param bitmap  bitmap
	 * @param oldRect Rect 对象表示一个矩形区域，从 (0,0) 到 (200,200) 之间的矩形区域
	 * @param newRect 将 mBitmap 的 (100,100) 到 (300,300) 区域拿出来，自动缩放并画到屏幕的 (100,100) 到 (200,200) 区域。
	 * @return bitmap
	 */
	public static Bitmap tailorBitmap(Bitmap bitmap, Rect oldRect, Rect newRect) {
		//图片的某个区域拿出来画到屏幕的指定区域
		Canvas canvas = new Canvas();
		canvas.drawBitmap(bitmap, oldRect, newRect, null);
		return bitmap;
	}


	/**
	 * convert drawable to bitmap
	 * @param drawable drawable
	 * @return bitmap
	 */
	public static Bitmap convertDrawableToBitmap(Drawable drawable) {
		//判断当前drawable是否不等于透明
		Bitmap.Config config = drawable.getOpacity() != PixelFormat.OPAQUE ? Bitmap.Config.ARGB_8888 : Bitmap.Config.RGB_565;
		Bitmap bitmap = Bitmap.createBitmap(drawable.getIntrinsicWidth(), drawable.getIntrinsicHeight(), config);
		//create a empty bitmap   width and height are specified
		Canvas canvas = new Canvas(bitmap);
		//set bounds for drawable , like a rect object , specified range,  this method usually called from draw method.
		drawable.setBounds(0, 0, drawable.getIntrinsicWidth(), drawable.getIntrinsicHeight());
		drawable.draw(canvas);
		return bitmap;
	}


	/**
	 * 将bitmap对象转换为drawable对象
	 * @param context Context
	 * @param bitmap  需要转换的bitmap对象
	 * @return drawable  转换后的drawable对象
	 */
	public static Drawable convertBitmapToDrawable(Context context, Bitmap bitmap) {
		Drawable drawable = new BitmapDrawable(context.getResources(), bitmap);
		return drawable;
	}


	/**
	 * 目前这个方法主要用来保存二维码扫描出来的bitmap对象
	 * @param context context
	 * @param bitmap  bitmap
	 * @param quality quality
	 * @return true -- save succ   false -- save failure
	 */
	public static Boolean saveBitmapInLocal(Context context, Bitmap bitmap, int quality) {
		String path = null;
		FileOutputStream out = null;
		try {

			if (Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
				path = context.getExternalCacheDir().getAbsolutePath() + "/scanImg";
			} else {
				path = context.getCacheDir().getAbsolutePath() + "/scanImg";
			}

			File file = new File(path);
			if (!file.exists()) {
				file.mkdirs();
			}

			SimpleDateFormat format = new SimpleDateFormat("yyyy_MM_dd_HH_mm_ss");
			String time = format.format(new Date());
			String fileName = time + ".jpg";

			path = path + "/" + fileName;

			File file2 = new File(path);


			if (!file2.exists()) {
				file2.createNewFile();
			} else {
				file2.delete();
			}

			out = new FileOutputStream(path);
			bitmap.compress(CompressFormat.JPEG, quality, out);
			out.flush();

			it("angle", getPicAngle(path));

		} catch (Exception e) {
			e.printStackTrace();
			return false;

		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		return true;
	}


	/**
	 * 目前这个方法用来保存上传图片时经过压缩后的bitmap对象 将bitmap保存在本地
	 * @param bitmap   bitmap
	 * @param quality  quality
	 * @param savePath savePath
	 * @return true -- save succ   false -- save failure
	 */
	public static Boolean saveBitmapInLocal2(Bitmap bitmap, int quality, String savePath) {

		FileOutputStream out = null;
		try {

			File file2 = new File(savePath);

			if (!file2.exists()) {
				file2.createNewFile();
			} else {
				file2.delete();
			}

			out = new FileOutputStream(savePath);
			bitmap.compress(CompressFormat.JPEG, quality, out);
			out.flush();

			it("saveBitmapInLocal2---angle", getPicAngle(savePath));

		} catch (Exception e) {
			e.printStackTrace();
			return false;

		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		return true;
	}


	/**
	 * 该方法有个缺陷  图片大小超过1M就会内存溢出!!!
	 * @param path     //图片真实路径
	 * @param savePath //图片将要被存储到的路径
	 * @param size     //压缩比率  建议为1-2中间
	 * @param quality  //压缩的质量
	 * @return true or false
	 * 将指定路径的图片压缩指定的比率到新的路径;
	 */
	public static boolean CompressLoacalBitmap(String path, String savePath, int size, int quality) {

		boolean brt = false;

		Options opts = new Options();
		opts.inJustDecodeBounds = true;
		BitmapFactory.decodeFile(path, opts);

		opts.inSampleSize = size;
		opts.inJustDecodeBounds = false;

		try {
			Bitmap bitMap = BitmapFactory.decodeFile(path, opts);

			int width = bitMap.getWidth();
			int height = bitMap.getHeight();

			float scaleWidth = (float) 0.8;
			float scaleHeight = (float) 0.8;

			int degree = BitmapUtil.getPicAngle(path);

			Matrix matrix = new Matrix();
			matrix.postScale(scaleWidth, scaleHeight);
			matrix.postRotate(degree);

			bitMap = Bitmap.createBitmap(bitMap, 0, 0, width, height, matrix, true);

			FileOutputStream out = new FileOutputStream(savePath);
			bitMap.compress(CompressFormat.JPEG, quality, out);

			out.flush();
			out.close();
			bitMap.recycle();

			brt = true;

		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return brt;

		} catch (OutOfMemoryError err) {

		}
		return brt;

	}


	/**
	 * 根据文件路径得到bitmap对象  AppSuggestActivity
	 * @param filepath  filepath
	 * @param reqWidth  所需图片压缩尺寸最小宽度
	 * @param reqHeight 所需图片压缩尺寸最小高度
	 * @return 返回经过文件路径处理得到的bitmap对象
	 */
	public static Bitmap decodeSampledBitmapFromFile(String filepath, int reqWidth, int reqHeight) {
		final Options options = new Options();
		options.inJustDecodeBounds = true;
		BitmapFactory.decodeFile(filepath, options);

		options.inSampleSize = calculateInSampleSize(options, reqWidth, reqHeight);
		options.inJustDecodeBounds = false;
		return BitmapFactory.decodeFile(filepath, options);
	}


	/**
	 * 得到drawable中指定图片的bitmap对象
	 * @param res       Context.getResource()
	 * @param resId     被转换的图片的id
	 * @param reqWidth  所需图片压缩尺寸最小宽度
	 * @param reqHeight 所需图片压缩尺寸最小高度
	 * @return 经过转换后的bitmap对象
	 */
	public static Bitmap decodeSampledBitmapFromResource(Resources res, int resId, int reqWidth, int reqHeight) {
		final Options options = new Options();
		options.inJustDecodeBounds = true;
		BitmapFactory.decodeResource(res, resId, options);

		options.inSampleSize = calculateInSampleSize(options, reqWidth, reqHeight);
		options.inJustDecodeBounds = false;
		return BitmapFactory.decodeResource(res, resId, options);
	}


	/**
	 * 将bitmap对象压缩到指定大小
	 * @param bitmap    被压缩的bitmap对象
	 * @param reqWidth  需要的宽度
	 * @param reqHeight 需要的高度
	 * @return 返回经过压缩后的bitmap对象
	 */
	public static Bitmap decodeSampledBitmapFromBitmap(Bitmap bitmap, int reqWidth, int reqHeight) {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		bitmap.compress(CompressFormat.PNG, 90, baos);
		byte[] data = baos.toByteArray();

		final Options options = new Options();
		options.inJustDecodeBounds = true;
		BitmapFactory.decodeByteArray(data, 0, data.length, options);
		options.inSampleSize = calculateInSampleSize(options, reqWidth, reqHeight);
		options.inJustDecodeBounds = false;
		return BitmapFactory.decodeByteArray(data, 0, data.length, options);
	}


	/**
	 * 计算当前图片压缩到制定大小需要的压缩比率
	 * 计算压缩比例值
	 * @param options   解析图片的配置信息
	 * @param reqWidth  所需图片压缩尺寸最小宽度
	 * @param reqHeight 所需图片压缩尺寸最小高度
	 * @return 合适的图片压缩比率
	 */
	public static int calculateInSampleSize(Options options, int reqWidth, int reqHeight) {

		final int picheight = options.outHeight;
		final int picwidth = options.outWidth;

		it("calculateInSampleSize", "原尺寸:" + picwidth + "---" + picheight + "倍");

		int targetheight = picheight;
		int targetwidth = picwidth;
		int inSampleSize = 1;

		if (targetheight > reqHeight || targetwidth > reqWidth) {
			while (targetheight >= reqHeight && targetwidth >= reqWidth) {
				it("calculateInSampleSize", "压缩:" + inSampleSize + "倍");
				inSampleSize += 1;
				targetheight = picheight / inSampleSize;
				targetwidth = picwidth / inSampleSize;
			}
		}
		it("calculateInSampleSize", "最终压缩比例:" + inSampleSize + "倍");
		it("calculateInSampleSize", "新尺寸" + targetwidth + "---" + targetheight);
		return inSampleSize;
	}


	/**
	 * 质量压缩
	 * @param image 被压缩的bitmap对象
	 * @param maxkb 被压缩后的最大大小 (KB)
	 * @return 被压缩后的bitmap对象
	 */
	public static Bitmap compressBitmap(Bitmap image, int maxkb) {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		// 质量压缩方法，这里100表示不压缩，把压缩后的数据存放到baos中
		image.compress(CompressFormat.JPEG, 100, baos);
		int options = 100;
		it("compressBitmap", "原始大小" + baos.toByteArray().length);
		// 循环判断如果压缩后图片是否大于(maxkb),大于继续压缩
		while (baos.toByteArray().length / 1024 > maxkb) {
			it("compressBitmap", "压缩一次!");
			// 重置baos即清空baos
			baos.reset();
			// 每次都减少10
			options -= 10;
			// 这里压缩options%，把压缩后的数据存放到baos中
			image.compress(CompressFormat.JPEG, options, baos);
		}
		it("compressBitmap", "压缩后大小" + baos.toByteArray().length);
		// 把压缩后的数据baos存放到ByteArrayInputStream中
		ByteArrayInputStream isBm = new ByteArrayInputStream(baos.toByteArray());
		// 把ByteArrayInputStream数据生成图片
		Bitmap bitmap = BitmapFactory.decodeStream(isBm, null, null);
		return bitmap;
	}


	/**
	 * 放大或缩小指定的bitmap
	 * @param source        被放大或缩小的bitmap
	 * @param requireWidth  指定放大或缩小的宽度
	 * @param requireHeight 指定放大或缩小的高度
	 * @return 被放大或缩小后的bitmap
	 */
	public static Bitmap enlargeOrZoomBitmap(Bitmap source, int requireWidth, int requireHeight) {
		int width = source.getWidth();
		int height = source.getHeight();
		Matrix matrix = new Matrix();
		float scaleWidth = ((float) requireWidth / width);
		float scaleHeight = ((float) requireHeight / height);
		matrix.postScale(scaleWidth, scaleHeight);
		return Bitmap.createBitmap(source, 0, 0, width, height, matrix, true);
	}

	private static void it(String tag, Object msg) {
		LogManager.getDefaultLogger().it(tag, msg);
	}


}
