package org.tuzhao.library.view;

import android.content.Context;
import android.support.v4.view.MotionEventCompat;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.view.MotionEvent;
import android.view.ViewConfiguration;
import android.view.WindowManager;
import android.widget.RelativeLayout;

/**
 * @author HY  the author
 * @author tuzhao the modifier
 */
public class TouchLayout extends RelativeLayout {

	public static final int DIRECTION_LEFT = 0;
	public static final int DIRECTION_RIGHT = 1;
	private static final int INVALID_POINTER = -1;

	private int mTouchSlop;
	private int minMoveX;
	private int maxMoveY;
	private int mActivePointerId = INVALID_POINTER;

	private float xPosition;
	private float yPosition;

	private boolean loadingFlag = false;

	private onSwitchListener onSwitchListener;

	public TouchLayout(Context context) {
		this(context, null);
	}

	public TouchLayout(Context context, AttributeSet attrs) {
		super(context, attrs);
		init(context);
	}

	public TouchLayout(Context context, AttributeSet attrs, int defStyle) {
		super(context, attrs, defStyle);
		init(context);
	}

	private void init(Context context) {
		mTouchSlop = ViewConfiguration.get(context).getScaledTouchSlop();
		WindowManager winMgr = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
		DisplayMetrics dm = new DisplayMetrics();
		winMgr.getDefaultDisplay().getMetrics(dm);
		minMoveX = dm.widthPixels / 4;
		maxMoveY = (dm.heightPixels / 8) * 3;
	}

	@Override
	public boolean dispatchTouchEvent(MotionEvent ev) {

		int action = MotionEventCompat.getActionMasked(ev);

		switch (action) {
			case MotionEvent.ACTION_DOWN:
				mActivePointerId = MotionEventCompat.getPointerId(ev, 0);

				final float initialDownX = getMotionEventX(ev, mActivePointerId);
				final float initialDownY = getMotionEventY(ev, mActivePointerId);

				if (initialDownX == -1 || initialDownY == -1) {
					return false;
				}

				if (xPosition == 0 || yPosition == 0) {
					xPosition = initialDownX;
					yPosition = initialDownY;

				}
				break;
			case MotionEvent.ACTION_MOVE:
				if (mActivePointerId == INVALID_POINTER) {
					return false;
				}

				final float tempX = getMotionEventX(ev, mActivePointerId);
				final float tempY = getMotionEventY(ev, mActivePointerId);

				if (tempX == -1 || tempY == -1) {
					return false;
				}

				if (Math.abs((tempY - yPosition)) >= mTouchSlop && Math.abs((tempX - xPosition)) >= mTouchSlop) {
					if (Math.abs((tempY - yPosition)) <= maxMoveY) {
						if (isCanMove()) {
							float xDisplacement = tempX - xPosition;
							if (xDisplacement > 0 & xDisplacement > minMoveX) {
								if (onSwitchListener != null) {
									onSwitchListener.onSwitch(DIRECTION_RIGHT);
								}
							}
							if (xDisplacement < 0 & (Math.abs(xDisplacement) > minMoveX)) {
								if (onSwitchListener != null) {
									onSwitchListener.onSwitch(DIRECTION_LEFT);
								}
							}
						}
					}
				}
				break;
			case MotionEvent.ACTION_POINTER_UP:
				onSecondaryPointerUp(ev);
				break;
			case MotionEvent.ACTION_UP:
			case MotionEvent.ACTION_CANCEL:
				mActivePointerId = INVALID_POINTER;
				if (xPosition != 0 & yPosition != 0) {
					xPosition = 0;
					yPosition = 0;
				}
				break;
			default:
				break;
		}
		return super.dispatchTouchEvent(ev);
	}

	@Override
	public boolean onTouchEvent(MotionEvent event) {
		return true;
	}

	private boolean isCanMove() {
		return !loadingFlag;
	}

	public interface onSwitchListener {
		void onSwitch(int direction);
	}

	/**
	 * @param listener onSwitchListener
	 */
	public void setOnSwitchListener(onSwitchListener listener) {
		this.onSwitchListener = listener;
	}

	/**
	 * if you want to load page one by one,you must use this method to set state.
	 * if you don`t set state , the onSwitchListener will call method onSwitch() every time.
	 * @param flag true the layout is loading page ,the method onSwitch() will not call;
	 *             false  the layout isn`t loading page or the page has been load complete.
	 */
	public void setLoadState(boolean flag) {
		xPosition = 0;
		yPosition = 0;
		loadingFlag = flag;
	}

	private float getMotionEventX(MotionEvent ev, int activePointerId) {
		final int index = MotionEventCompat.findPointerIndex(ev, activePointerId);
		if (index < 0) {
			return -1;
		}
		return MotionEventCompat.getX(ev, index);
	}

	private float getMotionEventY(MotionEvent ev, int activePointerId) {
		final int index = MotionEventCompat.findPointerIndex(ev, activePointerId);
		if (index < 0) {
			return -1;
		}
		return MotionEventCompat.getY(ev, index);
	}

	private void onSecondaryPointerUp(MotionEvent ev) {
		final int pointerIndex = MotionEventCompat.getActionIndex(ev);
		final int pointerId = MotionEventCompat.getPointerId(ev, pointerIndex);
		if (pointerId == mActivePointerId) {
			// This was our active pointer going up. Choose a new active pointer and adjust accordingly.
			final int newPointerIndex = pointerIndex == 0 ? 1 : 0;
			mActivePointerId = MotionEventCompat.getPointerId(ev, newPointerIndex);
		}
	}

}
