package org.tuzhao.library.recycler.base;

import android.app.Activity;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import org.tuzhao.library.debug.LogManager;
import org.tuzhao.library.tools.StringUtil;
import org.tuzhao.library.tools.TimeZoneUtil;

import java.util.List;

/**
 * 全局抽象的基础的 recycler adapter
 * @author tuzhao
 *         Created by tuzhao on 16-3-17.
 */
public abstract class BaseRecyclerAdapter<T extends BaseRecyclerViewHolder, D> extends RecyclerView.Adapter<T> implements View.OnClickListener {

	private List<D> dataList;
	private LayoutInflater inflater;
	private int layoutId;

	private OnItemClickListener listener;

	public BaseRecyclerAdapter(Activity context, List<D> dataList, int layoutId) {
		this.dataList = dataList;
		this.layoutId = layoutId;
		inflater = LayoutInflater.from(context);
	}

	@Override
	public T onCreateViewHolder(ViewGroup parent, int viewType) {
		View v = inflater.inflate(layoutId, parent, false);
		v.setOnClickListener(this);
		return youCreateViewHolder(v);
	}

	@Override
	public void onBindViewHolder(T holder, int position) {
		holder.setRootViewTag(dataList.get(position));
		youBindViewHolder(holder, position);
	}

	public abstract T youCreateViewHolder(View v);

	public abstract void youBindViewHolder(T holder, int position);

	@Override
	public int getItemCount() {
		return dataList == null ? 0 : dataList.size();
	}

	public D remove(int position) {
		D remove = dataList.remove(position);
		notifyItemRemoved(position);
		return remove;
	}

	public void add(D d, int position) {
		dataList.add(position, d);
		notifyItemInserted(position);
	}

	public List<D> getDataList() {
		return dataList;
	}

	@Override
	public void onClick(View v) {
		if (listener != null) {
			Object o = v.getTag();
			int position = -1;
			for (int i = 0; i < dataList.size(); i++) {
				if (o == dataList.get(i)) {
					position = i;
					break;
				}
			}
			listener.onItemClick(v, v.getTag(), position);
		}
	}

	public interface OnItemClickListener {
		void onItemClick(View v, Object data, int position);
	}

	public void setOnItemClickListener(OnItemClickListener listener) {
		this.listener = listener;
	}

	public abstract class ClickOneViewListener<E> implements View.OnClickListener {

		private long lastTimeMillis;
		private long interval = 1000;
		public E e;

		public ClickOneViewListener(E o) {
			this.e = o;
		}

		public ClickOneViewListener(E o, long interval) {
			this.e = o;
			this.interval = interval;
		}

		@Override
		public void onClick(View v) {
			if (isTimeEnabled()) {
				int position = -1;
				for (int i = 0; i < dataList.size(); i++) {
					if (e == dataList.get(i)) {
						position = i;
						break;
					}
				}
				click(v, position);
			}
		}

		public abstract void click(View v, int position);

		private boolean isTimeEnabled() {
			long currentTimeMillis = System.currentTimeMillis();
			if ((currentTimeMillis - lastTimeMillis) > interval) {
				lastTimeMillis = currentTimeMillis;
				return true;
			}
			return false;
		}

	}

	public String trans(String sourceTime) {
		return TimeZoneUtil.trans(sourceTime);
	}

	public String trans3(String sourceTime) {
		return TimeZoneUtil.trans3(sourceTime);
	}

	public String trans2(String sourceTime) {
		return TimeZoneUtil.trans2(sourceTime);
	}

	/**
	 * 将浮点数格式化为小数点后两位
	 * @param num double
	 * @return 保留两位小数的数字
	 */
	public String format(double num) {
		return StringUtil.formatPointTwo(num);
	}

	/**
	 * 获取中国时区当前的毫秒时间戳
	 * @return 中国时区当前的毫秒时间戳
	 */
	public long getChinaCurrentMils() {
		return TimeZoneUtil.getChinaCurrentMils();
	}

	/**
	 * 将浮点数格式化为整数
	 * @param num double
	 * @return 整数
	 */
	public String formatInteger(double num) {
		return StringUtil.formatInteger(num);
	}

	public void it(Object tag,Object msg){
		LogManager.getDefaultLogger().it(String.valueOf(tag),msg);
	}

}
