/*
 * Copyright 2016 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.uberfire.security.impl.authz;

import java.util.Collection;
import java.util.Iterator;
import java.util.TreeSet;

import org.jboss.errai.common.client.api.annotations.Portable;
import org.uberfire.security.authz.Permission;
import org.uberfire.security.authz.PermissionCollection;

import static org.uberfire.security.authz.AuthorizationResult.*;

/**
 * A collection where the permissions are ordered by name.
 */
@Portable
public class DefaultPermissionCollection implements PermissionCollection {

    private TreeSet<Permission> permissionSet = new TreeSet<>();

    public DefaultPermissionCollection() {
    }

    @Override
    public Collection<Permission> collection() {
        return permissionSet;
    }

    @Override
    public PermissionCollection add(Permission... permissions) {
        for (Permission p : permissions) {

            // Avoid redundancy
            if (!implies(p)) {
                permissionSet.add(p);
            }
        }
        return this;
    }

    @Override
    public PermissionCollection remove(Permission... permissions) {
        for (Permission p : permissions) {
            permissionSet.remove(p);
        }
        return this;
    }

    @Override
    public Permission get(String name) {
        for (Permission p : permissionSet) {
            if (equalsName(name, p.getName())) {
                return p;
            }
        }
        return null;
    }

    protected boolean equalsName(String s1, String s2) {
        return (s1 == null && s2 == null) || (s1 != null && s1.equals(s2));
    }

    @Override
    public boolean implies(Permission permission) {
        for (Permission p : permissionSet) {
            if (p.implies(permission)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean impliesName(Permission permission) {
        for (Permission p : permissionSet) {
            if (p.impliesName(permission)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public PermissionCollection merge(PermissionCollection other, int priority) {
        if (other == null || other.collection().isEmpty()) {
            return this;
        }

        DefaultPermissionCollection result = new DefaultPermissionCollection();

        for (Permission pOther : other.collection()) {
            addToCollectionIf(result, this, pOther, priority);
        }
        for (Permission pThis : this.collection()) {
            addToCollectionIf(result, other, pThis, priority*-1);
        }
        return result;
    }

    /**
     * Add the given permission to the result only when some of the following conditions are met:
     * <br/>
     * <br/> 1. The permission is not implied by name in the source collection</li>
     * <br/> 2. The permission is implied by name and the priority is positive</li>
     * <br/> 3. The permission is granted, implied by name and the priority doesn't count (= 0)
     *
     * @param result The collection where the permission shall be added
     * @param source The collection used to check conditions #2 & #3
     * @param p The permission to add to the result
     * @param priority integer indicating how to proceed in conditions #2 & #3
     * <ul>
     *     <li>0 = same priority (the permission is added only if GRANTED)</li>
     *     <li>negative integer = the permission is ruled out</li>
     *     <li>positive integer = the permission is added</li>
     * </ul>
     */
    private void addToCollectionIf(PermissionCollection result, PermissionCollection source, Permission p, int priority) {

        if (!source.impliesName(p) ||
            priority > 0 ||
            (priority == 0 && ACCESS_GRANTED.equals(p.getResult()))) {

            result.add(p);
        }
    }

    public DefaultPermissionCollection clone() {
        DefaultPermissionCollection clone = new DefaultPermissionCollection();
        for (Permission p : permissionSet) {
            clone.add(p.clone());
        }
        return clone;
    }

    public PermissionCollection invert(Permission target) {
        target.setResult(target.getResult().invert());

        // After inverting the permission ensure no implied permissions are left
        Iterator<Permission> it = permissionSet.iterator();
        while (it.hasNext()) {
            Permission p = it.next();
            if (!target.equals(p) && target.implies(p)) {
                it.remove();
            }
        }
        return this;
    }

    @Override
    public String toString() {
        StringBuilder out = new StringBuilder();
        Iterator<Permission> it = permissionSet.iterator();
        while (it.hasNext()) {
            Permission p = it.next();
            out.append(p).append("\n");
        }
        return out.toString();
    }
}
