package org.uma.jmetal.operator.mutation.impl;

import org.uma.jmetal.operator.mutation.MutationOperator;
import org.uma.jmetal.solution.doublesolution.DoubleSolution;
import org.uma.jmetal.solution.doublesolution.repairsolution.RepairDoubleSolution;
import org.uma.jmetal.solution.doublesolution.repairsolution.impl.RepairDoubleSolutionWithBoundValue;
import org.uma.jmetal.util.bounds.Bounds;
import org.uma.jmetal.util.errorchecking.JMetalException;
import org.uma.jmetal.util.pseudorandom.JMetalRandom;
import org.uma.jmetal.util.pseudorandom.RandomGenerator;

/**
 * This class implements a Lévy flight mutation operator for real-valued solutions. Lévy flights are
 * characterized by heavy-tailed distributions with infinite variance, producing mostly small steps
 * with occasional very large jumps. This behavior is beneficial for global optimization as it
 * provides both local search capabilities and the ability to escape local optima through large
 * jumps.
 *
 * <p>The implementation uses the Mantegna algorithm to generate Lévy-distributed steps: 1. Generate
 * u ~ Normal(0, σ_u²) where σ_u = [Γ(1+β)sin(πβ/2)/Γ((1+β)/2)β2^((β-1)/2)]^(1/β) 2. Generate v ~
 * Normal(0, 1) 3. Lévy step = u / |v|^(1/β)
 *
 * <p><b>Parameters:</b>
 * <ul>
 * <li><b>mutationProbability:</b> The probability of mutating each variable. Must be in [0, 1].</li>
 * <li><b>beta:</b> The Lévy index parameter. Must be in (1, 2]. Controls the tail heaviness:
 *     <ul>
 *     <li>Values closer to 1.0 produce heavier tails with more frequent large jumps</li>
 *     <li>Values around 1.5 provide balanced exploration with moderate large jumps (typical choice)</li>
 *     <li>Values closer to 2.0 approach Gaussian behavior with fewer large jumps</li>
 *     </ul>
 * </li>
 * <li><b>stepSize:</b> The scaling factor for Lévy steps. Must be positive (> 0).
 *     Typical values range from 0.001 to 0.1:
 *     <ul>
 *     <li>Smaller values (0.001-0.01) provide fine-grained local search</li>
 *     <li>Medium values (0.01-0.05) balance local and global search</li>
 *     <li>Larger values (0.05-0.1) emphasize global exploration</li>
 *     </ul>
 * </li>
 * </ul>
 *
 * @author Antonio J. Nebro
 *     <p>Code generated by Claude Sonnet 4
 */
public class LevyFlightMutation implements MutationOperator<DoubleSolution> {
  private static final double DEFAULT_BETA = 1.5;
  private static final double DEFAULT_STEP_SIZE = 0.01; 

  private double mutationProbability;
  private double beta; // Lévy index parameter (1 < beta <= 2)
  private double stepSize; // Scaling factor for Lévy steps (0.01 < stepSize < 1.0)  
  private RepairDoubleSolution solutionRepair;
  private RandomGenerator<Double> randomGenerator;

  /**
   * Constructor
   * @param mutationProbability
   */
  public LevyFlightMutation(double mutationProbability) {
  this(mutationProbability, DEFAULT_BETA, DEFAULT_STEP_SIZE);
} 

  /**
   * Constructor with default parameters Default: beta = 1.5 (typical for Lévy flights), stepSize =
   * 0.01
   */
  public LevyFlightMutation() {
    this(0.01, 1.5, 0.01);
  }

  /**
   * Constructor
   *
   * @param mutationProbability The probability of mutating each variable
   * @param beta The Lévy index parameter
   * @param stepSize The scaling factor for Lévy steps
   */
  public LevyFlightMutation(double mutationProbability, double beta, double stepSize) {
    this(mutationProbability, beta, stepSize, new RepairDoubleSolutionWithBoundValue());
  }

  /**
   * Constructor
   *
   * @param mutationProbability The probability of mutating each variable
   * @param beta The Lévy index parameter
   * @param stepSize The scaling factor for Lévy steps
   * @param solutionRepair The repair strategy for out-of-bounds values
   */
  public LevyFlightMutation(
          double mutationProbability,
          double beta,
          double stepSize,
          RepairDoubleSolution solutionRepair) {
    this(
            mutationProbability,
            beta,
            stepSize,
            solutionRepair,
            () -> JMetalRandom.getInstance().nextDouble());
  }

  /**
   * Constructor
   *
   * @param mutationProbability The probability of mutating each variable
   * @param beta The Lévy index parameter
   * @param stepSize The scaling factor for Lévy steps
   * @param solutionRepair The repair strategy for out-of-bounds values
   * @param randomGenerator The random number generator
   */
  public LevyFlightMutation(
          double mutationProbability,
          double beta,
          double stepSize,
          RepairDoubleSolution solutionRepair,
          RandomGenerator<Double> randomGenerator) {
    if (mutationProbability < 0 || mutationProbability > 1) {
      throw new JMetalException("Mutation probability must be in [0, 1]");
    }
    if (beta <= 1 || beta > 2) {
      throw new JMetalException("Beta parameter must be in (1, 2]");
    }
    if (stepSize <= 0) {
      throw new JMetalException("Step size must be positive");
    }

    this.mutationProbability = mutationProbability;
    this.beta = beta;
    this.stepSize = stepSize;
    this.solutionRepair = solutionRepair;
    this.randomGenerator = randomGenerator;
  }

  /**
   * Execute the mutation operation
   *
   * @param solution The solution to be mutated
   * @return The mutated solution
   */
  @Override
  public DoubleSolution execute(DoubleSolution solution) {
    if (solution == null) {
      throw new JMetalException("Null parameter");
    }

    doMutation(mutationProbability, solution);
    return solution;
  }

  /**
   * Perform the mutation operation on the solution
   *
   * @param probability The mutation probability
   * @param solution The solution to mutate
   */
  private void doMutation(double probability, DoubleSolution solution) {
    for (int i = 0; i < solution.variables().size(); i++) {
      if (randomGenerator.getRandomValue() <= probability) {
        double currentValue = solution.variables().get(i);
        Bounds<Double> bounds = solution.getBounds(i);
        double lowerBound = bounds.getLowerBound();
        double upperBound = bounds.getUpperBound();

        // Generate Lévy flight step
        double levyStep = generateLevyStep();

        // Scale step by variable range and step size parameter
        double perturbation = levyStep * stepSize * (upperBound - lowerBound);

        // Apply perturbation
        double newValue = currentValue + perturbation;

        // Repair the solution if it goes out of bounds
        newValue = solutionRepair.repairSolutionVariableValue(newValue, lowerBound, upperBound);

        solution.variables().set(i, newValue);
      }
    }
  }

  /**
   * Generate a Lévy-distributed step using the Mantegna algorithm
   *
   * @return A Lévy-distributed random step
   */
  private double generateLevyStep() {
    // Calculate sigma_u using the Mantegna algorithm
    double numerator = gamma(1 + beta) * Math.sin(Math.PI * beta / 2);
    double denominator = gamma((1 + beta) / 2) * beta * Math.pow(2, (beta - 1) / 2);
    double sigmaU = Math.pow(numerator / denominator, 1 / beta);

    // Generate u ~ Normal(0, sigma_u^2)
    double u = generateGaussian() * sigmaU;

    // Generate v ~ Normal(0, 1)
    double v = generateGaussian();

    // Calculate Lévy step: u / |v|^(1/beta)
    double levyStep = u / Math.pow(Math.abs(v), 1 / beta);

    return levyStep;
  }

  /**
   * Generate a Gaussian random number using Box-Muller transform
   *
   * @return A standard normal random number
   */
  private double generateGaussian() {
    // Box-Muller transformation
    double u1 = randomGenerator.getRandomValue();
    double u2 = randomGenerator.getRandomValue();

    // Avoid log(0)
    if (u1 < 1e-10) {
      u1 = 1e-10;
    }

    return Math.sqrt(-2 * Math.log(u1)) * Math.cos(2 * Math.PI * u2);
  }

  /**
   * Approximation of the gamma function using Stirling's approximation For the range of values used
   * in Lévy flights, this provides sufficient accuracy
   *
   * @param x The input value
   * @return Approximation of Γ(x)
   */
  private double gamma(double x) {
    if (x < 0) {
      throw new JMetalException("Gamma function not defined for negative values");
    }

    // Handle special cases
    if (Math.abs(x - 1.0) < 1e-10) {
      return 1.0; // Γ(1) = 1
    }
    if (Math.abs(x - 2.0) < 1e-10) {
      return 1.0; // Γ(2) = 1
    }
    if (Math.abs(x - 0.5) < 1e-10) {
      return Math.sqrt(Math.PI); // Γ(0.5) = √π
    }
    if (Math.abs(x - 1.5) < 1e-10) {
      return 0.5 * Math.sqrt(Math.PI); // Γ(1.5) = 0.5√π
    }

    // Use Stirling's approximation for other values
    // Γ(x) ≈ √(2π/x) * (x/e)^x
    if (x > 1) {
      return Math.sqrt(2 * Math.PI / x) * Math.pow(x / Math.E, x);
    } else {
      // For x < 1, use recurrence relation: Γ(x) = Γ(x+1) / x
      return gamma(x + 1) / x;
    }
  }

  /**
   * Get the mutation probability
   *
   * @return The mutation probability
   */
  @Override
  public double mutationProbability() {
    return mutationProbability;
  }

  /**
   * Set the mutation probability
   *
   * @param mutationProbability The new mutation probability
   */
  public void mutationProbability(double mutationProbability) {
    if (mutationProbability < 0 || mutationProbability > 1) {
      throw new JMetalException("Mutation probability must be in [0, 1]");
    }
    this.mutationProbability = mutationProbability;
  }

  /**
   * Get the beta parameter
   *
   * @return The beta parameter
   */
  public double beta() {
    return beta;
  }

  /**
   * Set the beta parameter
   *
   * @param beta The new beta parameter
   */
  public void beta(double beta) {
    if (beta <= 1 || beta > 2) {
      throw new JMetalException("Beta parameter must be in (1, 2]");
    }
    this.beta = beta;
  }

  /**
   * Get the step size parameter
   *
   * @return The step size parameter
   */
  public double stepSize() {
    return stepSize;
  }

  /**
   * Set the step size parameter
   *
   * @param stepSize The new step size parameter
   */
  public void stepSize(double stepSize) {
    if (stepSize <= 0) {
      throw new JMetalException("Step size must be positive");
    }
    this.stepSize = stepSize;
  }

  /**
   * Get the solution repair strategy
   *
   * @return The solution repair strategy
   */
  public RepairDoubleSolution solutionRepair() {
    return solutionRepair;
  }

  /**
   * Set the solution repair strategy
   *
   * @param solutionRepair The new solution repair strategy
   */
  public void solutionRepair(RepairDoubleSolution solutionRepair) {
    this.solutionRepair = solutionRepair;
  }

  /**
   * Get string representation of the operator
   *
   * @return String representation
   */
  @Override
  public String toString() {
    return "LevyFlightMutation{"
        + "mutationProbability="
        + mutationProbability
        + ", beta="
        + beta
        + ", stepSize="
        + stepSize
        + '}';
  }
}
