package org.uma.jmetal.operator.mutation.impl;

import org.uma.jmetal.operator.mutation.MutationOperator;
import org.uma.jmetal.solution.doublesolution.DoubleSolution;
import org.uma.jmetal.solution.doublesolution.repairsolution.RepairDoubleSolution;
import org.uma.jmetal.solution.doublesolution.repairsolution.impl.RepairDoubleSolutionWithBoundValue;
import org.uma.jmetal.util.bounds.Bounds;
import org.uma.jmetal.util.errorchecking.JMetalException;
import org.uma.jmetal.util.pseudorandom.JMetalRandom;
import org.uma.jmetal.util.pseudorandom.RandomGenerator;

/**
 * This class implements a power-law mutation operator for real-valued solutions. The power-law
 * distribution produces heavy-tailed perturbations that can occasionally create large jumps while
 * favoring smaller perturbations, which is beneficial for both exploration and exploitation in
 * optimization.
 *
 * <p>The mutation follows the formula: tempDelta = rnd^(-delta) deltaq = 0.5 * (rnd - 0.5) * (1 -
 * tempDelta) newValue = oldValue + deltaq * (upperBound - lowerBound)
 *
 * <p><b>Parameters:</b>
 * <ul>
 * <li><b>mutationProbability:</b> The probability of mutating each variable. Must be in [0, 1].</li>
 * <li><b>delta:</b> The power-law exponent parameter (controls distribution shape).
 *     Must be positive (> 0). Typical values range from 0.1 to 10.0:
 *     <ul>
 *     <li>Values < 1.0 create more uniform distributions with moderate perturbations</li>
 *     <li>Values around 1.0 provide balanced exploration/exploitation</li>
 *     <li>Values > 1.0 create heavy-tailed distributions favoring small perturbations
 *         with occasional large jumps</li>
 *     </ul>
 * </li>
 * </ul>
 *
 * Code generated by Claude Sonnet 4
 *
 * @author Antonio J. Nebro.
 */
public class PowerLawMutation implements MutationOperator<DoubleSolution> {
  private static final double DEFAULT_DELTA = 1.0;
  private static final double DEFAULT_MUTATION_PROBABILITY = 0.01;

  private double mutationProbability;
  private double delta;
  private RepairDoubleSolution solutionRepair;
  private RandomGenerator<Double> randomGenerator;

  /** Constructor with default parameters */
  public PowerLawMutation() {
    this(DEFAULT_MUTATION_PROBABILITY, DEFAULT_DELTA);
  }

  /** Constructor */
  public PowerLawMutation(double mutationProbability) {
    this(mutationProbability, DEFAULT_DELTA);
  }

  /**
   * Constructor
   *
   * @param mutationProbability The probability of mutating each variable
   * @param delta The power-law exponent parameter (controls distribution shape)
   */
  public PowerLawMutation(double mutationProbability, double delta) {
    this(mutationProbability, delta, new RepairDoubleSolutionWithBoundValue());
  }

  /**
   * Constructor
   *
   * @param mutationProbability The probability of mutating each variable
   * @param delta The power-law exponent parameter (controls distribution shape)
   * @param solutionRepair The repair strategy for out-of-bounds values
   */
  public PowerLawMutation(
          double mutationProbability, double delta, RepairDoubleSolution solutionRepair) {
    this(mutationProbability, delta, solutionRepair, () -> JMetalRandom.getInstance().nextDouble());
  }

  /**
   * Constructor
   *
   * @param mutationProbability The probability of mutating each variable
   * @param delta The power-law exponent parameter (controls distribution shape)
   * @param solutionRepair The repair strategy for out-of-bounds values
   * @param randomGenerator The random number generator
   */
  public PowerLawMutation(
          double mutationProbability,
          double delta,
          RepairDoubleSolution solutionRepair,
          RandomGenerator<Double> randomGenerator) {
    if (mutationProbability < 0 || mutationProbability > 1) {
      throw new JMetalException("Mutation probability must be in [0, 1]");
    }
    if (delta <= 0) {
      throw new JMetalException("Delta parameter must be positive");
    }

    this.mutationProbability = mutationProbability;
    this.delta = delta;
    this.solutionRepair = solutionRepair;
    this.randomGenerator = randomGenerator;
  }

  /**
   * Execute the mutation operation
   *
   * @param solution The solution to be mutated
   * @return The mutated solution
   */
  @Override
  public DoubleSolution execute(DoubleSolution solution) {
    if (solution == null) {
      throw new JMetalException("Null parameter");
    }

    doMutation(mutationProbability, solution);
    return solution;
  }

  /**
   * Perform the mutation operation on the solution
   *
   * @param probability The mutation probability
   * @param solution The solution to mutate
   */
  private void doMutation(double probability, DoubleSolution solution) {
    for (int i = 0; i < solution.variables().size(); i++) {
      if (randomGenerator.getRandomValue() <= probability) {
        double currentValue = solution.variables().get(i);
        Bounds<Double> bounds = solution.getBounds(i);
        double lowerBound = bounds.getLowerBound();
        double upperBound = bounds.getUpperBound();

        // Generate power-law distributed perturbation
        double rnd = randomGenerator.getRandomValue();

        // Avoid division by zero or extreme values
        if (rnd < 1e-10) {
          rnd = 1e-10;
        } else if (rnd > 1 - 1e-10) {
          rnd = 1 - 1e-10;
        }

        double tempDelta = Math.pow(rnd, -delta);
        double deltaq = 0.5 * (rnd - 0.5) * (1 - tempDelta);

        // Apply perturbation scaled by variable range
        double newValue = currentValue + deltaq * (upperBound - lowerBound);

        // Repair the solution if it goes out of bounds
        newValue = solutionRepair.repairSolutionVariableValue(newValue, lowerBound, upperBound);

        solution.variables().set(i, newValue);
      }
    }
  }

  /**
   * Get the mutation probability
   *
   * @return The mutation probability
   */
  @Override
  public double mutationProbability() {
    return mutationProbability;
  }

  /**
   * Set the mutation probability
   *
   * @param mutationProbability The new mutation probability
   */
  public void mutationProbability(double mutationProbability) {
    if (mutationProbability < 0 || mutationProbability > 1) {
      throw new JMetalException("Mutation probability must be in [0, 1]");
    }
    this.mutationProbability = mutationProbability;
  }

  /**
   * Get the delta parameter
   *
   * @return The delta parameter
   */
  public double delta() {
    return delta;
  }

  /**
   * Set the delta parameter
   *
   * @param delta The new delta parameter
   */
  public void delta(double delta) {
    if (delta <= 0) {
      throw new JMetalException("Delta parameter must be positive");
    }
    this.delta = delta;
  }

  /**
   * Get the solution repair strategy
   *
   * @return The solution repair strategy
   */
  public RepairDoubleSolution solutionRepair() {
    return solutionRepair;
  }

  /**
   * Set the solution repair strategy
   *
   * @param solutionRepair The new solution repair strategy
   */
  public void solutionRepair(RepairDoubleSolution solutionRepair) {
    this.solutionRepair = solutionRepair;
  }

  /**
   * Get string representation of the operator
   *
   * @return String representation
   */
  @Override
  public String toString() {
    return "PowerLawMutation{"
        + "mutationProbability="
        + mutationProbability
        + ", delta="
        + delta
        + '}';
  }
}
