package org.unidal.maven.plugin.source.lines.handler;

import org.unidal.maven.plugin.source.lines.entity.ClassModel;
import org.unidal.maven.plugin.source.lines.entity.CountModel;
import org.unidal.maven.plugin.source.lines.entity.ProjectModel;
import org.unidal.maven.plugin.source.lines.entity.RootModel;
import org.unidal.maven.plugin.source.lines.transform.BaseVisitor;
import org.unidal.maven.plugin.source.pipeline.Source;
import org.unidal.maven.plugin.source.pipeline.SourceHandlerAdaptor;
import org.unidal.maven.plugin.source.pipeline.SourceHandlerContext;
import org.unidal.maven.plugin.source.pipeline.SourceScope;

public class LineCounter extends SourceHandlerAdaptor {
	private RootModel m_root = new RootModel();

	@Override
	public void handleEnd(SourceHandlerContext ctx, SourceScope scope) {
		if (scope.isProject()) {
			m_root.accept(new CountAggregator());
			m_root.accept(new CountPrinter());

			System.out.println(m_root);
		}

		ctx.fireEnd(scope);
	}

	@Override
	public void handleStart(SourceHandlerContext ctx, SourceScope scope) {
		if (scope.isFile()) {
			Source source = ctx.source();
			SourceScope parentScope = source.getParentScope();

			if (parentScope.isSource()) {
				ProjectModel project = m_root.findOrCreateProject(source.getModule().getName());
				ClassModel klass = project.findOrCreateClass(source.getPackage(), source.getClassName());

				if (parentScope.isTest()) {
					klass.setTest(true);
				}

				if (source.getSourceRoot().contains("/target/generated-")) {
					klass.setGenerated(true);
				}
			}

		}

		ctx.fireStart(scope);
	}

	@Override
	public void handleLine(SourceHandlerContext ctx, String line) {
		Source source = ctx.source();
		SourceScope parentScope = source.getParentScope();

		if (parentScope.isSource()) {
			ProjectModel project = m_root.findOrCreateProject(source.getModule().getName());
			ClassModel klass = project.findOrCreateClass(source.getPackage(), source.getClassName());
			CountModel count = klass.getCount();

			if (count == null) {
				count = new CountModel();
				klass.setCount(count);
			}

			String str = line.trim();

			if (str.isEmpty()) {
				count.incEmpty();
			} else if (str.startsWith("//") || str.startsWith("/*")) {
				count.incComment();

				if (str.contains("/* THIS FILE WAS AUTO GENERATED BY codegen-maven-plugin, DO NOT EDIT IT */")) {
					klass.setGenerated(true);
				}
			}

			count.incLines();
		}
	}

	private static class CountAggregator extends BaseVisitor {
		private CountModel m_root;

		private CountModel m_project;

		@Override
		public void visitCount(CountModel count) {
			m_root.add(count);
			m_project.add(count);
		}

		@Override
		public void visitClass(ClassModel _class) {
			CountModel count = _class.getCount();

			count.incFiles();

			if (_class.isGenerated()) {
				count.incGeneratedFiles();
			}

			if (_class.isTest()) {
				count.incTestFiles();
			}

			super.visitClass(_class);
		}

		@Override
		public void visitProject(ProjectModel project) {
			m_project = new CountModel();

			super.visitProject(project);
			project.setCount(m_project);
		}

		@Override
		public void visitRoot(RootModel root) {
			m_root = new CountModel();

			super.visitRoot(root);
			root.setCount(m_root);
		}
	}

	private static class CountPrinter extends BaseVisitor {
		@Override
		public void visitProject(ProjectModel project) {
			int len = "<?xml version=\"1.0\" encoding=\"utf-8\"?>".length();

			System.out.println(project.getName() + ": " + project.getCount().toString().substring(len));
		}

		@Override
		public void visitRoot(RootModel root) {
			super.visitRoot(root);

			int len = "<?xml version=\"1.0\" encoding=\"utf-8\"?>".length();

			System.out.println("TOTAL: " + root.getCount().toString().substring(len));
		}
	}

}
