package org.unitils.ftp;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Properties;
import java.util.Set;

import org.mockftpserver.fake.FakeFtpServer;
import org.mockftpserver.fake.UserAccount;
import org.mockftpserver.fake.filesystem.DirectoryEntry;
import org.mockftpserver.fake.filesystem.FileEntry;
import org.mockftpserver.fake.filesystem.FileSystem;
import org.mockftpserver.fake.filesystem.UnixFakeFileSystem;
import org.mockftpserver.fake.filesystem.WindowsFakeFileSystem;
import org.unitils.core.Module;
import org.unitils.core.TestListener;
import org.unitils.core.UnitilsException;
import org.unitils.ftp.annotations.TestFtpServer;
import org.unitils.thirdparty.org.apache.commons.io.FileUtils;
import org.unitils.util.AnnotationUtils;
import org.unitils.util.PropertyUtils;
import org.unitils.util.ReflectionUtils;


/**
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * 
 * @since 1.0.0
 */
public class FtpModule implements Module {

    private File defaultBaseFolder;

    private String os;

    private String encoding;

    private String ftpPort;

    private String userName;

    private String password;
    
    protected static final String PACKAGENAME = "org.unitils.ftp.";
    
    /**The key for the username in unitils.properties     */
    public static final String PROP_USERNAME = PACKAGENAME + "username";
    /**The key for the basefolder in unitils.properties     */
    public static final String PROP_BASEFOLDER = PACKAGENAME +  "baseFolder";
    /**The key for the password in unitils.properties     */
    public static final String PROP_PASSWORD = PACKAGENAME + "password";
    /**The key for the encoding in unitils.properties     */
    public static final String PROP_ENCODING = PACKAGENAME + "fileEncoding";
    /**The key for the port in unitils.properties     */
    public static final String PROP_PORT = PACKAGENAME + "port";

    @Override
    public void init(Properties configuration) {

        defaultBaseFolder = new File(configuration.getProperty(PROP_BASEFOLDER));
        os = configuration.getProperty("sun.desktop");


        userName = configuration.getProperty(PROP_USERNAME);
        password = configuration.getProperty(PROP_PASSWORD);
        encoding = PropertyUtils.getString(PROP_ENCODING, "UTF-8", configuration);
        ftpPort = PropertyUtils.getString(PROP_PORT, "21", configuration);


    }

    /**
     * Creates a {@link FakeFtpServer} with the properties out of unitils.properties and adds all the files to the server.
     * @param field
     * @return FakeFtpServer
     * @throws IOException
     */
    private FakeFtpServer initServer(Field field) throws IOException {
        FakeFtpServer server = new FakeFtpServer();
        UserAccount account = new UserAccount();

        account.setUsername(userName);
        account.setPassword(password);

        server.setServerControlPort(Integer.valueOf(ftpPort));
        server.addUserAccount(account);

        TestFtpServer annotation = field.getAnnotation(TestFtpServer.class);
        File baseFolder = defaultBaseFolder;
        if (!annotation.baseFolder().isEmpty()) {
            baseFolder = new File(annotation.baseFolder());
        }
        FileSystem fileSystem = null;
        if ("WINDOWS".equalsIgnoreCase(os)) {
            fileSystem = new WindowsFakeFileSystem();
        } else {
            fileSystem = new UnixFakeFileSystem();
        }


        addToFileSystem(fileSystem, baseFolder);
        server.setFileSystem(fileSystem);
        account.setHomeDirectory(baseFolder.getAbsolutePath());

        server.start();
        return server;
    }

    /**
     * If file is a file than it is added to the {@link FileSystem}.
     * If the file is a directory than the directory is added and this method is called to add the children.
     * @param fileSystem
     * @param file
     * @throws IOException
     */
    private void addToFileSystem(FileSystem fileSystem, File file) throws IOException {
        if (file.isFile()) {
            FileEntry entry = new FileEntry(file.getAbsolutePath(), FileUtils.readFileToString(file, encoding));
            fileSystem.add(entry);
        }
        if (file.isDirectory()) {
            DirectoryEntry entry = new DirectoryEntry(file.getAbsolutePath());
            fileSystem.add(entry);
            for (File child : file.listFiles()) {
                addToFileSystem(fileSystem, child);
            }
        }
    }

    @Override
    public void afterInit() {
        //do nothing
    }

    @Override
    public TestListener getTestListener() {
        return new FtpTestListener();
    }


    private class FtpTestListener extends TestListener {

        private FakeFtpServer server;

        @Override
        public void beforeTestSetUp(Object testObject, Method testMethod) {
            try {
                Field field = getServerField(testObject);
                if (field != null) {
                    server = initServer(field);
                    ReflectionUtils.setFieldValue(testObject, field, server);
                }
            } catch (IOException e) {
                throw new UnitilsException("Initializing FtpModule failed ", e);
            }
        }


        private Field getServerField(Object testObject) {
            Set<Field> fieldSet = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), TestFtpServer.class);
            if (fieldSet.isEmpty()) {
                return null;
            }
            Field field = fieldSet.iterator().next();
            return field;
        }

        @Override
        public void afterTestMethod(Object testObject, Method testMethod, Throwable testThrowable) {
            if (server != null) {
                server.stop();
            }
        }


    }

}
