package org.unitils.jbehave.core;

import static org.jbehave.core.annotations.AfterScenario.Outcome.ANY;
import static org.jbehave.core.annotations.AfterScenario.Outcome.FAILURE;
import static org.jbehave.core.annotations.AfterScenario.Outcome.SUCCESS;
import static org.jbehave.core.steps.StepType.GIVEN;
import static org.jbehave.core.steps.StepType.THEN;
import static org.jbehave.core.steps.StepType.WHEN;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.jbehave.core.annotations.AfterScenario;
import org.jbehave.core.annotations.AfterScenario.Outcome;
import org.jbehave.core.annotations.AfterStory;
import org.jbehave.core.annotations.Alias;
import org.jbehave.core.annotations.Aliases;
import org.jbehave.core.annotations.BeforeScenario;
import org.jbehave.core.annotations.BeforeStory;
import org.jbehave.core.annotations.Composite;
import org.jbehave.core.annotations.Given;
import org.jbehave.core.annotations.ScenarioType;
import org.jbehave.core.annotations.Then;
import org.jbehave.core.annotations.When;
import org.jbehave.core.configuration.Configuration;
import org.jbehave.core.steps.BeforeOrAfterStep;
import org.jbehave.core.steps.InjectableStepsFactory;
import org.jbehave.core.steps.PatternVariantBuilder;
import org.jbehave.core.steps.StepCandidate;
import org.jbehave.core.steps.StepCollector.Stage;
import org.jbehave.core.steps.StepCreator;
import org.jbehave.core.steps.StepType;
import org.jbehave.core.steps.Steps;
import org.unitils.jbehave.core.stepcreator.UnitilsStepCreator;
import org.unitils.jbehave.core.steps.BasicUnitilsSteps;
import org.unitils.jbehave.core.steps.UnitilsStepsFactory;
import org.unitils.util.AnnotationUtils;
import org.unitils.util.ReflectionUtils;


/**
 * Extended {@link Steps}: This class will add the correct steps to the runner.
 *  
 * @author Willemijn Wouters
 * 
 * @since 1.0.0
 * 
 */
public class UnitilsSteps extends Steps {

    /**
     * @param configuration
     * @param type
     * @param stepsFactory
     */
    public UnitilsSteps(Configuration configuration, Class<?> type, InjectableStepsFactory stepsFactory) {
        super(configuration, type, stepsFactory);

    }

    /**
     * @see org.jbehave.core.steps.Steps#listBeforeOrAfterScenario(org.jbehave.core.annotations.ScenarioType)
     */
    @Override
    public List<BeforeOrAfterStep> listBeforeOrAfterScenario(ScenarioType type) {
        List<BeforeOrAfterStep> steps = new ArrayList<BeforeOrAfterStep>();
        StepCreator stepCreator = createStepCreator();
        InjectableStepsFactory stepsFactory = getStepsFactory();

        steps.addAll(UnitilsStepsFactory.createBeforeTestClassStep(stepCreator, stepsFactory, type()));
        steps.addAll(UnitilsStepsFactory.createAfterCreateTestObject(stepCreator, stepsFactory, type()));
        steps.addAll(UnitilsStepsFactory.createBeforeTestSetUp(stepCreator, stepsFactory, type()));

        steps.addAll(scenarioStepsHaving(type, Stage.BEFORE, BeforeScenario.class));
        steps.addAll(UnitilsStepsFactory.createBeforeTestMethod(stepCreator, stepsFactory, type()));
        steps.addAll(UnitilsStepsFactory.createAfterTestMethod(stepCreator, stepsFactory, type()));
        steps.addAll(scenarioStepsHaving(type, Stage.AFTER, AfterScenario.class, ANY, SUCCESS, FAILURE));
        steps.addAll(UnitilsStepsFactory.createAfterTestTearDown(stepCreator, stepsFactory, type()));
        return Collections.unmodifiableList(steps);
    }

    /**
     * @see org.jbehave.core.steps.Steps#listBeforeOrAfterStory(boolean)
     */
    @Override
    public List<BeforeOrAfterStep> listBeforeOrAfterStory(boolean givenStory) {
        List<BeforeOrAfterStep> steps = new ArrayList<BeforeOrAfterStep>();
        steps.addAll(stepsHaving(Stage.BEFORE, BeforeStory.class, givenStory));
        steps.addAll(stepsHaving(Stage.AFTER, AfterStory.class, givenStory));
        return Collections.unmodifiableList(steps);
    }

    /**
     * Creates a list of scenarios by Stage.
     * @param type
     * @param stage
     * @param annotationClass
     * @param outcomes
     * @return {@link List}
     */
    protected List<BeforeOrAfterStep> scenarioStepsHaving(ScenarioType type, Stage stage,
        Class<? extends Annotation> annotationClass, Outcome... outcomes) {
        List<BeforeOrAfterStep> steps = new ArrayList<BeforeOrAfterStep>();
        for (Method method : AnnotationUtils.getMethodsAnnotatedWith(type(), annotationClass)) {
            ScenarioType scenarioType = scenarioType(method, annotationClass);
            if (type == scenarioType) {
                if (stage == Stage.BEFORE) {
                    steps.add(createBeforeOrAfterStep(stage, method));
                }
                if (stage == Stage.AFTER) {
                    Outcome scenarioOutcome = scenarioOutcome(method, annotationClass);
                    for (Outcome outcome : outcomes) {
                        if (outcome.equals(scenarioOutcome)) {
                            steps.add(createBeforeOrAfterStep(stage, method, outcome));
                        }
                    }
                }
            }
        }
        return steps;
    }

    /**
     * Creates a {@link List} with {@link BeforeOrAfterStep}s with methods who use a specific {@link Annotation}.
     * @param stage
     * @param annotationClass
     * @param givenStory
     * @return {@link List}
     */
    protected List<BeforeOrAfterStep> stepsHaving(Stage stage, Class<? extends Annotation> annotationClass,
        boolean givenStory) {
        List<BeforeOrAfterStep> steps = new ArrayList<BeforeOrAfterStep>();
        for (final Method method : AnnotationUtils.getMethodsAnnotatedWith(type(), annotationClass)) {
            if (runnableStoryStep(method.getAnnotation(annotationClass), givenStory)) {
                steps.add(createBeforeOrAfterStep(stage, method));
            }
        }
        return steps;
    }

    /**
     * Check if an annotation is a runnable story step.
     * @param annotation
     * @param givenStory
     * @return {@link Boolean}
     */
    protected boolean runnableStoryStep(Annotation annotation, boolean givenStory) {
        boolean uponGivenStory = uponGivenStory(annotation);
        return uponGivenStory == givenStory;
    }

    /**
     * check the value of the {@link BeforeStory#uponGivenStory()} or {@link AfterStory#uponGivenStory()}.
     * @param annotation
     * @return boolean
     */
    protected boolean uponGivenStory(Annotation annotation) {
        if (annotation instanceof BeforeStory) {
            return ((BeforeStory) annotation).uponGivenStory();
        } else if (annotation instanceof AfterStory) {
            return ((AfterStory) annotation).uponGivenStory();
        }
        return false;
    }
    
    /**
     * This methods gets the {@link ScenarioType} out of the {@link BeforeScenario} or {@link AfterScenario}.
     * @param method
     * @param annotationClass
     * @return {@link ScenarioType}
     */
    protected ScenarioType scenarioType(Method method, Class<? extends Annotation> annotationClass) {
        if (annotationClass.isAssignableFrom(BeforeScenario.class)) {
            return ((BeforeScenario) method.getAnnotation(annotationClass)).uponType();
        }
        if (annotationClass.isAssignableFrom(AfterScenario.class)) {
            return ((AfterScenario) method.getAnnotation(annotationClass)).uponType();
        }
        return ScenarioType.NORMAL;
    }

    /**
     * This method creates a basic {@link BeforeOrAfterStep}.
     * @param stage
     * @param method
     * @return {@link BeforeOrAfterStep}
     */
    protected BeforeOrAfterStep createBeforeOrAfterStep(Stage stage, Method method) {
        return createBeforeOrAfterStep(stage, method, Outcome.ANY);
    }

    /**
     * This method creates a basic {@link BeforeOrAfterStep}.
     * @param stage
     * @param method
     * @param outcome
     * @return {@link BeforeOrAfterStep}
     */
    protected BeforeOrAfterStep createBeforeOrAfterStep(Stage stage, Method method, Outcome outcome) {
        return new BasicUnitilsSteps(stage, method, outcome, createStepCreator());
    }

    /**
     * Get the {@link org.jbehave.core.annotations.AfterScenario.Outcome} out of the {@link AfterScenario}.
     * @param method
     * @param annotationClass
     * @return
     */
    protected Outcome scenarioOutcome(Method method, Class<? extends Annotation> annotationClass) {
        if (annotationClass.isAssignableFrom(AfterScenario.class)) {
            return ((AfterScenario) method.getAnnotation(annotationClass)).uponOutcome();
        }
        return Outcome.ANY;
    }

    /**
     * This method creates a new {@link UnitilsStepCandidate}.
     * @param method
     * @param stepType
     * @param stepPatternAsString
     * @param priority
     * @param configuration
     * @return {@link StepCandidate}
     */
    protected StepCandidate createCandidate(Method method, StepType stepType, String stepPatternAsString, int priority, Configuration configuration) {
        return new UnitilsStepCandidate(stepPatternAsString, priority, stepType, method, type(), getStepsFactory(),
            configuration.keywords(), configuration.stepPatternParser(), configuration.parameterConverters(), configuration.parameterControls());
    }

    /**
     * This method makes a new {@link StepCandidate} and adds it to the list of candidates.
     * @param candidates
     * @param method
     * @param stepType
     * @param stepPatternAsString
     * @param priority
     */
    protected void addCandidate(List<StepCandidate> candidates, Method method, StepType stepType, String stepPatternAsString, int priority) {
        checkForDuplicateCandidates(candidates, stepType, stepPatternAsString);
        StepCandidate candidate = createCandidate(method, stepType, stepPatternAsString, priority, configuration());
        candidate.useStepMonitor(configuration().stepMonitor());
        candidate.useParanamer(configuration().paranamer());
        candidate.doDryRun(configuration().storyControls().dryRun());
        if (method.isAnnotationPresent(Composite.class)) {
            candidate.composedOf(method.getAnnotation(Composite.class).steps());
        }
        candidates.add(candidate);
    }

    /**
     * Adds new {@link StepCandidate}s.
     * @param candidates
     * @param method
     * @param stepType
     * @param value
     * @param priority
     */
    protected void addCandidatesFromVariants(List<StepCandidate> candidates, Method method, StepType stepType, String value, int priority) {
        PatternVariantBuilder b = new PatternVariantBuilder(value);
        for (String variant : b.allVariants()) {
            addCandidate(candidates, method, stepType, variant, priority);
        }
    }

    /**
     * Adds new {@link StepCandidate}s.
     * @param candidates
     * @param method
     * @param stepType
     * @param priority
     */
    protected void addCandidatesFromAliases(List<StepCandidate> candidates, Method method, StepType stepType, int priority) {
        if (method.isAnnotationPresent(Aliases.class)) {
            String[] aliases = method.getAnnotation(Aliases.class).values();
            for (String alias : aliases) {
                addCandidatesFromVariants(candidates, method, stepType, alias, priority);
            }
        }
        if (method.isAnnotationPresent(Alias.class)) {
            String alias = method.getAnnotation(Alias.class).value();
            addCandidatesFromVariants(candidates, method, stepType, alias, priority);
        }
    }
    /**
     * checks for duplicate {@link StepCandidate}s.
     * @param candidates
     * @param stepType
     * @param patternAsString
     */
    protected void checkForDuplicateCandidates(List<StepCandidate> candidates, StepType stepType, String patternAsString) {
        for (StepCandidate candidate : candidates) {
            if (candidate.getStepType() == stepType && candidate.getPatternAsString().equals(patternAsString)) {
                throw new DuplicateCandidateFound(stepType, patternAsString);
            }
        }
    }

    /**
     * This method creates all the {@link StepCandidate}s for a specific type.
     * @see org.jbehave.core.steps.Steps#listCandidates()
     */
    public List<StepCandidate> listCandidates() {
        List<StepCandidate> candidates = new ArrayList<StepCandidate>();
        for (Method method : Arrays.asList(type().getMethods())) {
            if (method.isAnnotationPresent(Given.class)) {
                Given annotation = method.getAnnotation(Given.class);
                String value = annotation.value();
                int priority = annotation.priority();
                addCandidatesFromVariants(candidates, method, GIVEN, value, priority);
                addCandidatesFromAliases(candidates, method, GIVEN, priority);
            }
            if (method.isAnnotationPresent(When.class)) {
                When annotation = method.getAnnotation(When.class);
                String value = annotation.value();
                int priority = annotation.priority();
                addCandidatesFromVariants(candidates, method, WHEN, value, priority);
                addCandidatesFromAliases(candidates, method, WHEN, priority);
            }
            if (method.isAnnotationPresent(Then.class)) {
                Then annotation = method.getAnnotation(Then.class);
                String value = annotation.value();
                int priority = annotation.priority();
                addCandidatesFromVariants(candidates, method, THEN, value, priority);
                addCandidatesFromAliases(candidates, method, THEN, priority);
            }
        }
        return candidates;
    }

    /**
     * This method creates a new {@link UnitilsStepCreator}.
     * @return {@link StepCreator}
     */
    public StepCreator createStepCreator() {
        return new UnitilsStepCreator(type(), getStepsFactory() ,configuration().parameterConverters(), configuration().parameterControls(), null, configuration().stepMonitor());
    }

    /**
     * This is a getter for the stepsFactory.
     * @return {@link InjectableStepsFactory}
     */
    public InjectableStepsFactory getStepsFactory() {
        Field fieldStepsFactory = ReflectionUtils.getFieldWithName(Steps.class, "stepsFactory", false);
        return ReflectionUtils.getFieldValue(this, fieldStepsFactory);
    }
}
