package org.unitils.jbehave.core.reporters;

import com.google.common.base.Strings;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jbehave.core.model.Meta;
import org.jbehave.core.reporters.NullStoryReporter;
import org.unitils.core.UnitilsException;
import org.unitils.jbehave.modules.JBehaveTestLinkModule;

/**
 * Reporter which will make a connection with TestLink and update the testlink case with the scenario outcome.
 * This only happens in case the TestLink Unitils module is enabled and the executed scenario has the meta property @TestLinkId.
 *
 *
 * @author Suzan Rogge
 * @since 31/10/2014
 */
public class TestLinkJBehaveReporter extends NullStoryReporter {

    private static final Log LOGGER = LogFactory.getLog(TestLinkJBehaveReporter.class);
    private JBehaveTestLinkModule module = new JBehaveTestLinkModule();

    private Throwable failure;

    @Override
    public void beforeScenario(String s) {
        failure = null;
    }

    @Override
    public void failed(String s, Throwable throwable) {
        failure = throwable.getCause();
    }

    @Override
    public void scenarioMeta(Meta meta) {
        if (module.isEnabled()) {
            initializeCaseId(meta.getProperty("TestLinkId"));
            initializeSuiteId(meta.getProperty("TestLinkSuiteId"));
        }
    }

    @Override
    public void afterScenario() {
        if (module.isEnabled() && module.hasCaseId()) {
            try {
                module.updateTestCase(failure);
                LOGGER.info("testlink updated with result");
            } catch (UnitilsException e) {
                LOGGER.error("failed to update testlink: " + e.getMessage());
            }
        }
        module.reset();
    }

    private void initializeSuiteId(String suiteId) {
        if (suiteId == null) {
            suiteId = "";
        }
        module.setSuiteId(suiteId);
        LOGGER.info("testlink suite id " + suiteId + " initialized");
    }

    private void initializeCaseId(String caseId) {
        if (Strings.isNullOrEmpty(caseId)) {
            return;
        }
        try {
            module.setCaseId(caseId);
            LOGGER.info("testlink case id " + caseId + " initialized");
        } catch (Exception exception) {
            LOGGER.error("error initializing testlink case id: " + caseId + " for reason: " + exception.getMessage());
        }
    }

}
