package org.unitils.jbehave.core.stepcreator;

import static org.jbehave.core.steps.AbstractStepResult.failed;
import static org.jbehave.core.steps.AbstractStepResult.skipped;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;

import org.jbehave.core.annotations.Named;
import org.jbehave.core.annotations.AfterScenario.Outcome;
import org.jbehave.core.configuration.Keywords;
import org.jbehave.core.failures.BeforeOrAfterFailed;
import org.jbehave.core.failures.UUIDExceptionWrapper;
import org.jbehave.core.model.Meta;
import org.jbehave.core.parsers.StepMatcher;
import org.jbehave.core.steps.InjectableStepsFactory;
import org.jbehave.core.steps.ParameterControls;
import org.jbehave.core.steps.ParameterConverters;
import org.jbehave.core.steps.Step;
import org.jbehave.core.steps.StepCreator;
import org.jbehave.core.steps.StepMonitor;
import org.jbehave.core.steps.StepResult;
import org.unitils.jbehave.core.steps.IUnitilsStep;
import org.unitils.util.ReflectionUtils;

import com.thoughtworks.paranamer.Paranamer;


/**
 * Unitils Steps are invoked a little bit different, so the {@link UnitilsStepCreator} is a little bit modified.
 * 
 * @author Willemijn Wouters
 * 
 * @since 1.0.0
 * 
 */
public class UnitilsStepCreator extends StepCreator {
    /**
     * @param stepsType
     * @param stepsFactory
     * @param parameterConverters
     * @param parameterControls
     * @param stepMatcher
     * @param stepMonitor
     */
    public UnitilsStepCreator(Class<?> stepsType, InjectableStepsFactory stepsFactory, ParameterConverters parameterConverters, ParameterControls parameterControls, StepMatcher stepMatcher, StepMonitor stepMonitor) {
        super(stepsType, stepsFactory, parameterConverters, parameterControls, stepMatcher, stepMonitor);
    }
    
    /**
     * @see org.jbehave.core.steps.StepCreator#createBeforeOrAfterStep(java.lang.reflect.Method, org.jbehave.core.model.Meta)
     */
    @Override
    public Step createBeforeOrAfterStep(Method method, Meta meta) {
        return new StepCreatorBeforeOrAfterStep(method, meta);
    }
    
    /**
     * Returns the {@link ParameterName} representations for the method,
     * providing an abstraction that supports both annotated and non-annotated
     * parameters.
     * 
     * @param method the Method
     * @return The array of {@link ParameterName}s
     */
    protected ParameterName[] parameterNames(Method method) {
        String[] annotatedNames = annotatedParameterNames(method);
        String[] paranamerNames = paranamerParameterNames(method);

        ParameterName[] parameterNames = new ParameterName[annotatedNames.length];
        for (int i = 0; i < annotatedNames.length; i++) {
            parameterNames[i] = parameterName(annotatedNames, paranamerNames, i);
        }
        return parameterNames;
    }
    
    /**
     * This method checks if a parameter is annotated or not and transforms this information into a {@link ParameterName}.
     * @param annotatedNames
     * @param paranamerNames
     * @param i
     * @return {@link ParameterName}
     */
    protected ParameterName parameterName(String[] annotatedNames, String[] paranamerNames, int i) {
        String name = annotatedNames[i];
        boolean annotated = true;
        if (name == null) {
            name = (paranamerNames.length > i ? paranamerNames[i] : null);
            annotated = false;
        }
        return new ParameterName(name, annotated);
    }
    
    /**
     * @param method
     * @param outcome
     * @param storyAndScenarioMeta
     * @return {@link Step}
     * @see org.jbehave.core.steps.StepCreator#createAfterStepUponOutcome(java.lang.reflect.Method, org.jbehave.core.annotations.AfterScenario.Outcome, org.jbehave.core.model.Meta)
     */
    @Override
    public Step createAfterStepUponOutcome(Method method, Outcome outcome, Meta storyAndScenarioMeta) {
        switch (outcome) {
            case ANY:
            default:
                return new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta);
            case SUCCESS:
                return new UnitilsUponSuccessStep(method, storyAndScenarioMeta);
            case FAILURE:
                return new UnitilsFailureStep(method, storyAndScenarioMeta);
            }
    }
    
    /**
     * Creates a new {@link Step}
     * @see org.jbehave.core.steps.StepCreator#createAfterStepUponOutcome(java.lang.reflect.Method, org.jbehave.core.annotations.AfterScenario.Outcome, org.jbehave.core.model.Meta)
     * @param method
     * @param outcome
     * @param storyAndScenarioMeta
     * @param step
     * @return {@link Step}
     */
    public Step createAfterStepUponOutcome(Method method, Outcome outcome, Meta storyAndScenarioMeta, org.jbehave.core.steps.BeforeOrAfterStep step) {
        switch (outcome) {
            case ANY:
            default:
                return new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta, step);
            case SUCCESS:
                return new UnitilsUponSuccessStep(method, storyAndScenarioMeta, step);
            case FAILURE:
                return new UnitilsFailureStep(method, storyAndScenarioMeta, step);
            }
    }
    /**
     * 
     * In {@link org.jbehave.core.annotations.AfterScenario.Outcome} you can define when this scenario step should be invoked.
     * This method will only run when there is a previous step that fails.
     *
     */
    protected class UnitilsFailureStep extends UponFailureStep {
        private StepCreatorBeforeOrAfterStep beforeOrAfterStep;
        /**
         * @param method
         * @param storyAndScenarioMeta
         * @param beforeOrAfterStep
         */
        public UnitilsFailureStep(Method method, Meta storyAndScenarioMeta, org.jbehave.core.steps.BeforeOrAfterStep beforeOrAfterStep) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta, beforeOrAfterStep);
        }
        
        /**
         * @param method
         * @param storyAndScenarioMeta
         */
        public UnitilsFailureStep(Method method, Meta storyAndScenarioMeta) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta);
        }
        
        @Override
        public StepResult doNotPerform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return beforeOrAfterStep.perform(storyFailureIfItHappened);
        }
        @Override
        public StepResult perform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return skipped();
        }
        @Override
        public String asString(Keywords keywords) {
            return beforeOrAfterStep.asString(keywords);
        }
        
    }
    
    /**
     * 
     * In {@link org.jbehave.core.annotations.AfterScenario.Outcome} you can define when this scenario step should be invoked.
     * This method will only run when there is a previous step that succeeds.
     *
     */
    protected class UnitilsUponSuccessStep extends UponSuccessStep {
        private StepCreatorBeforeOrAfterStep beforeOrAfterStep;
        /**
         * @param method
         * @param storyAndScenarioMeta
         * @param step
         */
        public UnitilsUponSuccessStep(Method method, Meta storyAndScenarioMeta, org.jbehave.core.steps.BeforeOrAfterStep step) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta, step);
        }
        /**
         * @param method
         * @param storyAndScenarioMeta
         */
        public UnitilsUponSuccessStep(Method method, Meta storyAndScenarioMeta) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta);
        }
        
        @Override
        public StepResult doNotPerform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return skipped();
        }
        @Override
        public StepResult perform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return beforeOrAfterStep.perform(storyFailureIfItHappened);
        }
        @Override
        public String asString(Keywords keywords) {
            return beforeOrAfterStep.asString(keywords);
        }
    }
    
   
    
    /**
     * Extract parameter names using
     * {@link Paranamer#lookupParameterNames(AccessibleObject, boolean)}
     * 
     * @param method the Method inspected by Paranamer
     * @return An array of parameter names looked up by Paranamer
     */
    protected String[] paranamerParameterNames(Method method) {
        return getParanamer().lookupParameterNames(method, false);
    }
    
    /**
     * Extract parameter names using {@link Named}-annotated parameters
     * 
     * @param method the Method with {@link Named}-annotated parameters
     * @return An array of annotated parameter names, which <b>may</b> include
     *         <code>null</code> values for parameters that are not annotated
     */
    protected String[] annotatedParameterNames(Method method) {
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();
        String[] names = new String[parameterAnnotations.length];
        for (int i = 0; i < parameterAnnotations.length; i++) {
            for (Annotation annotation : parameterAnnotations[i]) {
                names[i] = annotationName(annotation);
            }
        }
        return names;
    }
    
    /**
     * Returns either the value of the annotation, either {@link Named} or
     * "javax.inject.Named".
     * 
     * @param annotation the Annotation
     * @return The annotated value or <code>null</code> if no annotation is
     *         found
     */
    protected String annotationName(Annotation annotation) {
        if (annotation.annotationType().isAssignableFrom(Named.class)) {
            return ((Named) annotation).value();
        } else if ("javax.inject.Named".equals(annotation.annotationType().getName())) {
            return Jsr330Helper.getNamedValue(annotation);
        } else {
            return null;
        }
    }
    
    /**
     * This is a different class, because the @Inject jar may not be in the
     * classpath.
     */
    public static class Jsr330Helper {

        public static String getNamedValue(Annotation annotation) {
            return ((Named) annotation).value();
        }

    }
    /**
     * 
     * This class is just like a BeforeOrAFterStep in the {@link StepCreator}.
     * But before
     *
     */
    public class StepCreatorBeforeOrAfterStep extends AbstractStep {

        
        private final Method method;
        private final Meta meta;
        private IUnitilsStep unitilsStep;

        public StepCreatorBeforeOrAfterStep(Method method, Meta meta) {
            this.method = method;
            this.meta = meta;
        }
        
        public StepCreatorBeforeOrAfterStep(Method method, Meta meta, org.jbehave.core.steps.BeforeOrAfterStep step) {
            this(method, meta);
            if (step instanceof IUnitilsStep) {
                this.unitilsStep = (IUnitilsStep) step;
            }
        }
        /**
         * @see org.jbehave.core.steps.Step#perform(org.jbehave.core.failures.UUIDExceptionWrapper)
         */
        @Override
        public StepResult perform(UUIDExceptionWrapper storyFailureIfItHappened) {
            ParameterConverters paramConvertersWithExceptionInjector = paramConvertersWithExceptionInjector(storyFailureIfItHappened);
            MethodInvoker methodInvoker = new MethodInvoker(method, paramConvertersWithExceptionInjector, getParanamer(), meta);

            try {
                if (unitilsStep != null) {
                    unitilsStep.invokeUnitilsMethod();
                } else {
                    methodInvoker.invoke();
                }
            } catch (InvocationTargetException e) {
                return failed(method, new UUIDExceptionWrapper(new BeforeOrAfterFailed(method, e.getCause())));
            } catch (Throwable t) {
                return failed(method, new UUIDExceptionWrapper(new BeforeOrAfterFailed(method, t)));
            }

            return skipped();
        }

       /**
        * Add a new {@link org.jbehave.core.steps.ParameterConverters.ParameterConverter}
        * @param storyFailureIfItHappened
        * @return {@link ParameterConverters}
        */
        protected ParameterConverters paramConvertersWithExceptionInjector(UUIDExceptionWrapper storyFailureIfItHappened) {
            return getParameterConverters().newInstanceAdding(new UUIDExceptionWrapperInjector(storyFailureIfItHappened));
        }

        /**
         * Do not perform {@link UUIDExceptionWrapper}
         * @see org.jbehave.core.steps.Step#doNotPerform(org.jbehave.core.failures.UUIDExceptionWrapper)
         */
        public StepResult doNotPerform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return perform(storyFailureIfItHappened);
        }
        
        /**
         * 
         * {@link org.jbehave.core.steps.ParameterConverters.ParameterConverter} for UUIDExceptions.
         *
         */
        public class UUIDExceptionWrapperInjector implements ParameterConverters.ParameterConverter {
            private final UUIDExceptionWrapper storyFailureIfItHappened;

            public UUIDExceptionWrapperInjector(UUIDExceptionWrapper storyFailureIfItHappened) {
                this.storyFailureIfItHappened = storyFailureIfItHappened;
            }

            /**
             * @see org.jbehave.core.steps.ParameterConverters.ParameterConverter#accept(java.lang.reflect.Type)
             */
            public boolean accept(Type type) {
                return UUIDExceptionWrapper.class == type;
            }

            public Object convertValue(String value, Type type) {
                return storyFailureIfItHappened;
            }
        }
        
        
        /**
         * @return the meta
         */
        public Meta getMeta() {
            return meta;
        }
        
        
        /**
         * @return the method
         */
        public Method getMethod() {
            return method;
        }
        
    }
    /**
     * 
     * invokes methods (@see MethodInvoker from {@link StepCreator}  (private class))
     *
     */
    protected class MethodInvoker {
        private final Method method;
        private final ParameterConverters parameterConverters;
        private final Paranamer paranamer;
        private final Meta meta;
        private int methodArity;

        public MethodInvoker(Method method, ParameterConverters parameterConverters, Paranamer paranamer, Meta meta) {
            this.method = method;
            this.parameterConverters = parameterConverters;
            this.paranamer = paranamer;
            this.meta = meta;
            this.methodArity = method.getParameterTypes().length;
        }

        /**
         * invoke the method field.
         * @throws InvocationTargetException
         * @throws IllegalAccessException
         */
        public void invoke() throws InvocationTargetException, IllegalAccessException {
            method.invoke(stepsInstance(), parameterValuesFrom(meta));
        }

        /**
         * All the parameters of a method.
         * @return array of {@link Parameter}
         */
        protected Parameter[] methodParameters() {
            Parameter[] parameters = new Parameter[methodArity];
            String[] annotationNamedParameters = annotatedParameterNames(method);
            String[] parameterNames = paranamer.lookupParameterNames(method, false);
            Class<?>[] parameterTypes = method.getParameterTypes();

            for (int paramPosition = 0; paramPosition < methodArity; paramPosition++) {
                String paramName = parameterNameFor(paramPosition, annotationNamedParameters, parameterNames);
                parameters[paramPosition] = new Parameter(paramPosition, parameterTypes[paramPosition], paramName);
            }

            return parameters;
        }

        /**
         * Get the name of a specific parameter.
         * @param paramPosition
         * @param annotationNamedParameters
         * @param parameterNames
         * @return {@link String}
         */
        protected String parameterNameFor(int paramPosition, String[] annotationNamedParameters, String[] parameterNames) {
            String nameFromAnnotation = nameIfValidPositionInArray(annotationNamedParameters, paramPosition);
            String parameterName = nameIfValidPositionInArray(parameterNames, paramPosition);
            if (nameFromAnnotation != null) {
                return nameFromAnnotation;
            } else if (parameterName != null) {
                return parameterName;
            }
            return null;
        }

        /**
         * Get the name of a specific parameter on a specified position.
         * @param paramNames
         * @param paramPosition
         * @return {@link String}
         */
        protected String nameIfValidPositionInArray(String[] paramNames, int paramPosition) {
            return paramPosition < paramNames.length ? paramNames[paramPosition] : null;
        }

        /**
         * Get the parameter values from the {@link Meta} data.
         * @param meta
         * @return array of {@link Object}s
         */
        protected Object[] parameterValuesFrom(Meta meta) {
            Object[] values = new Object[methodArity];
            for (Parameter parameter : methodParameters()) {
                values[parameter.position] = parameterConverters.convert(parameter.valueFrom(meta), parameter.type);
            }
            return values;
        }

        /**
         * Represents a parameter in a method.
         */
        protected class Parameter {
            private final int position;
            private final Class<?> type;
            private final String name;

            public Parameter(int position, Class<?> type, String name) {
                this.position = position;
                this.type = type;
                this.name = name;
            }

            public String valueFrom(Meta meta) {
                if (name == null) {
                    return null;
                }
                return meta.getProperty(name);
            }
        }
    }
    /**
     * Parameter: name of the parameter + is there an annotation.
     */
    public static class ParameterName {
        private String name;
        private boolean annotated;

        protected ParameterName(String name, boolean annotated) {
            this.name = name;
            this.annotated = annotated;
        }
        
        
        /**
         * @return the name
         */
        public String getName() {
            return name;
        }
        
        
        /**
         * @return the annotated
         */
        public boolean isAnnotated() {
            return annotated;
        }
    }
    
    
    /**
     * getter for field parameterConverters.
     * @return {@link ParameterConverters}
     */
    protected ParameterConverters getParameterConverters() {
        Field field = ReflectionUtils.getFieldWithName(StepCreator.class, "parameterConverters", false);
        return ReflectionUtils.getFieldValue(this, field);
    }
    
    /**
     * getter for field paranamer.
     * @return {@link Paranamer}
     */
    protected Paranamer getParanamer() {
        Field fieldParanamer = ReflectionUtils.getFieldWithName(StepCreator.class, "paranamer", false);
        return ReflectionUtils.getFieldValue(this, fieldParanamer);
    }
}
