/*
 * Copyright (c) Smals
 */
package org.unitils.jbehave;

import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.jbehave.core.configuration.Keywords;
import org.jbehave.core.reporters.Format;
import org.jbehave.core.reporters.StoryReporter;
import org.unitils.core.Unitils;
import org.unitils.jbehave.annotations.MetaFilters;
import org.unitils.jbehave.core.reporters.SeleniumScreenshotReporter;
import org.unitils.jbehave.core.reporters.SeleniumSteps;
import org.unitils.jbehave.steps.DatasetSupportSteps;
import org.unitils.util.AnnotationUtils;
import org.unitils.util.PropertyUtils;

import com.google.common.collect.Lists;


public class JBehaveConfiguration {
    
    private static final String UNITILS_META_FILTERS = "org.unitils.jbehave.meta_filters";

    private List<Object> steps;
    private List<String> metaFilters;
    private List<StoryReporter> reporters;
    private List<Format> formats;
    private Map<String, String> keywords;
    
    private String storyPackage;
    private String storyFile;
    private String screenshotDirectory;
    
    public JBehaveConfiguration() {
        steps = Lists.newArrayList();
        metaFilters = initializeMetaFilters();
        reporters = Lists.newArrayList();
        formats = Lists.newArrayList(Format.CONSOLE);
        storyPackage = "*";
        storyFile = "*.story";
        screenshotDirectory = "screenshot";
        keywords = Keywords.defaultKeywords();
    }
    
    private List<String> initializeMetaFilters() {
        List<String> metaFilters = Lists.newArrayList();
        MetaFilters annotationMetaFilters = AnnotationUtils.getClassLevelAnnotation(MetaFilters.class, this.getClass());
        //Check if the annotation is present on the testclass
        if (annotationMetaFilters != null) {
            metaFilters.add(annotationMetaFilters.value());
        } else {
            //if there is no annotation present on the test class, than the module must check it there is a default value in the unitils.properties
            Properties unitilsProperties = Unitils.getInstance().getConfiguration();
            if (PropertyUtils.containsProperty(UNITILS_META_FILTERS, unitilsProperties)) {
                metaFilters.add(PropertyUtils.getString(UNITILS_META_FILTERS, unitilsProperties));
            }
        }
        return metaFilters;
    }
    
    /**
     * Add support for @Dataset, @SqlScript and @ExpectedDataset in the story files.
     * @SqlScripts are always executed before @Dataset.
     */
    public JBehaveConfiguration useDatabaseSteps() {
        steps.add(new DatasetSupportSteps());
        return this;
    }
    
    public JBehaveConfiguration addSteps(Object stepsObject) {
        steps.add(stepsObject);
        return this;
    }
    
    /**
     * Add steps class which implements SeleniumSteps and register a selenium screenshot reporter
     * which takes a screenshot when a step fails.
     * Screenshots will be saved in given directory within target folder.
     */
    public JBehaveConfiguration addStepsWithSeleniumReporter(SeleniumSteps seleniumSteps, String directory) {
        steps.add(seleniumSteps);
        reporters.add(new SeleniumScreenshotReporter(directory, seleniumSteps));
        return this;
    }
    
    /**
     * Add steps class which implements SeleniumSteps and register a selenium screenshot reporter
     * which takes a screenshot when a step fails.
     * Screenshots are saved in default directory (target/screenshot.
     * Attention: when using this method, all screenshots of same failing step will be saved in same folder 
     * and therefore overwrite previous screenshot. To avoid this from happening, 
     *  use method addStepsWithSeleniumReporter(SeleniumSteps, String)
     *  or useDefaultScreenshotDirectory(String)
     * where you can specify a custom directory in which the screenshot is saved.
     */
    public JBehaveConfiguration addStepsWithSeleniumReporter(SeleniumSteps seleniumSteps) {
        return addStepsWithSeleniumReporter(seleniumSteps, screenshotDirectory);
    }
    
    public JBehaveConfiguration useDefaultScreenshotDirectory(String screenshotDirectory) {
        this.screenshotDirectory = screenshotDirectory;
        return this;
    }
    
    public JBehaveConfiguration addStoryReporter(StoryReporter storyReporter) {
        reporters.add(storyReporter);
        return this;
    }
    
    public JBehaveConfiguration addKeywordAlias(String keyword, String alias) {
        keywords.put(keyword, keywords.get(keyword) + "|" + alias);
        return this;
    }
    
    /**
     * add an additional meta filter.
     */
    public JBehaveConfiguration addMetaFilter(String metaFilter) {
        metaFilters.add(metaFilter);
        return this;
    }
    
    /**
     * Overwrite default meta filters.
     */
    public JBehaveConfiguration useMetaFilters(List<String> metaFilters) {
        this.metaFilters = metaFilters;
        return this;
    }
    
    public JBehaveConfiguration addFormat(Format format) {
        formats.add(format);
        return this;
    }
    
    public JBehaveConfiguration storyPackage(String storyPackage) {
        this.storyPackage = storyPackage;
        return this;
    }
    
    public JBehaveConfiguration storyFile(String storyFile) {
        this.storyFile = storyFile;
        return this;
    }
    
    public List<Object> steps() {
        return steps;
    }
    
    public List<String> metaFilters() {
        return metaFilters;
    }
    
    public StoryReporter[] reporters() {
        return reporters.toArray(new StoryReporter[reporters.size()]);
    }
    
    public Format[] formats() {
        return formats.toArray(new Format[formats.size()]);
    }
    
    public String storyFile() {
        return storyFile;
    }
    
    public String storyPackage() {
        return storyPackage;
    }
    
    public Keywords keywords() {
        return new Keywords(keywords);
    }
}

