package org.unitils.jbehave;

import java.io.File;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;

import org.jbehave.core.configuration.Configuration;
import org.jbehave.core.configuration.Keywords;
import org.jbehave.core.configuration.MostUsefulConfiguration;
import org.jbehave.core.failures.FailingUponPendingStep;
import org.jbehave.core.model.StoryMaps;
import org.jbehave.core.reporters.FreemarkerViewGenerator;
import org.jbehave.core.reporters.StoryReporterBuilder;
import org.jbehave.core.steps.MarkUnmatchedStepsAsPending;
import org.unitils.core.Factory;
import org.unitils.core.util.PropertiesReader;
import org.unitils.jbehave.core.parsers.RegexIncludingVariablesStoryParser;

/**
 * This class creates the {@link Configuration} for JBehave.
 * 
 * @author Willemijn Wouters
 * 
 * @since 1.0.0
 *
 */
public class JBehaveMostUseFulConfiguration implements Factory<Configuration> {

    private Logger logger = Logger.getLogger(this.getClass().getSimpleName());
    
    private JBehaveConfiguration jbehaveConfiguration;
    
    public JBehaveMostUseFulConfiguration(JBehaveConfiguration jbehaveConfiguration) {
        this.jbehaveConfiguration = jbehaveConfiguration;
    }
    
    /**
     * create a {@link MostUsefulConfiguration}.
     *
     * @return {@link Configuration}
     * @see org.unitils.core.Factory#create()
     */
    @Override
    public Configuration create() {
        MostUsefulConfiguration config = new MostUsefulConfiguration();
        Keywords keywords = jbehaveConfiguration.keywords();
        
        Properties viewResources = new Properties();
        viewResources.setProperty("views", "ftl/unitils-views.ftl");
        viewResources.setProperty("reports", "ftl/unitils-reports-with-totals.ftl");
        viewResources.setProperty("decorated", "ftl/unitils-report-decorated.ftl");
        
        StoryReporterBuilder storyReporterBuilder = new StoryReporterBuilder()
            .withViewResources(viewResources)
            .withKeywords(keywords)
            .withFormats(jbehaveConfiguration.formats())
            .withReporters(jbehaveConfiguration.reporters())
            .withFailureTrace(true)
            .withMultiThreading(false);
        
        PropertiesReader propertiesReader = new PropertiesReader();
        Properties jbehaveVariables = propertiesReader.loadPropertiesFileFromClasspath("jbehave.properties");
        
        return config
            .useKeywords(keywords)
            .useViewGenerator(new NoFailureViewGenerator())
            .useStoryReporterBuilder(storyReporterBuilder)
            .usePendingStepStrategy(new FailingUponPendingStep())
            .useStepCollector(new MarkUnmatchedStepsAsPending(keywords))
            .useStoryParser(new RegexIncludingVariablesStoryParser(jbehaveVariables, keywords));
    }
    
    /**
     * FreemarkerViewGenerator that will catch any exception thrown during the view generation.
     */
    public class NoFailureViewGenerator extends FreemarkerViewGenerator {
        
        @Override
        public void generateMapsView(File outputDirectory, StoryMaps storyMaps, Properties viewProperties) {
            try {
                super.generateMapsView(outputDirectory, storyMaps, viewProperties);
            } catch (Throwable throwable) {
                logger.warning("failed to generate maps view: " + throwable.getMessage());
            }
        }

        @Override
        public void generateReportsView(File outputDirectory, List<String> formats, Properties viewProperties) {
            try {
                super.generateReportsView(outputDirectory, formats, viewProperties);
            } catch (Throwable throwable) {
                logger.warning("failed to generate reports view: " + throwable.getMessage());
            }
        }
    }

}
