package org.unitils.jbehave.core;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.jbehave.core.annotations.AsParameterConverter;
import org.jbehave.core.configuration.Configuration;
import org.jbehave.core.steps.CandidateSteps;
import org.jbehave.core.steps.InstanceStepsFactory;
import org.jbehave.core.steps.ParameterConverters.MethodReturningConverter;
import org.jbehave.core.steps.ParameterConverters.ParameterConverter;
import org.unitils.util.AnnotationUtils;


/**
 * This class should use the {@link UnitilsSteps} instead of the ordinary {@link org.jbehave.core.steps.Steps}.
 *
 * @author Willemijn Wouters
 *
 * @since 1.0.0
 *
 */
public class UnitilsInstanceStepsFactory extends InstanceStepsFactory {

    private Configuration configuration;

    /**
     * @param configuration : the configuration for JBehave.
     * @param stepsInstances : all the steps that are used in the test.
     */
    public UnitilsInstanceStepsFactory(Configuration configuration, List<?> stepsInstances) {
        super(configuration, stepsInstances);
        this.configuration = configuration;
    }


    /**
     * Creates a list of {@link CandidateSteps}.
     * 
     * @return {@link List}
     * @see org.jbehave.core.steps.AbstractStepsFactory#createCandidateSteps()
     */
    @Override
    public List<CandidateSteps> createCandidateSteps() {
        List<Class<?>> types = stepsTypes();
        List<CandidateSteps> steps = new ArrayList<CandidateSteps>();
        for (Class<?> type : types) {
            configuration.parameterConverters().addConverters(methodReturningConverters(type));
            steps.add(new UnitilsSteps(configuration, type, this));
        }
        return steps;
    }

    /**
     * This method gets all the methods who are annotated with {@link AsParameterConverter}.
     *
     * @param type : type of the steps instance.
     * @return {@link List}
     */
    protected List<ParameterConverter> methodReturningConverters(Class<?> type) {
        List<ParameterConverter> converters = new ArrayList<ParameterConverter>();

        for (Method method : AnnotationUtils.getMethodsAnnotatedWith(type, AsParameterConverter.class)) {
            converters.add(new MethodReturningConverter(method, type, this));
        }

        return converters;
    }

}
