package org.unitils.jbehave.core;

import static org.jbehave.core.annotations.AfterScenario.Outcome.ANY;
import static org.jbehave.core.annotations.AfterScenario.Outcome.FAILURE;
import static org.jbehave.core.annotations.AfterScenario.Outcome.SUCCESS;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.jbehave.core.annotations.AfterScenario;
import org.jbehave.core.annotations.AfterScenario.Outcome;
import org.jbehave.core.annotations.BeforeScenario;
import org.jbehave.core.annotations.ScenarioType;
import org.jbehave.core.configuration.Configuration;
import org.jbehave.core.steps.BeforeOrAfterStep;
import org.jbehave.core.steps.InjectableStepsFactory;
import org.jbehave.core.steps.StepCandidate;
import org.jbehave.core.steps.StepCollector.Stage;
import org.jbehave.core.steps.StepCreator;
import org.jbehave.core.steps.StepType;
import org.jbehave.core.steps.Steps;
import org.unitils.jbehave.core.stepcreator.UnitilsStepCreator;
import org.unitils.jbehave.core.steps.BasicUnitilsSteps;
import org.unitils.jbehave.core.steps.UnitilsStepsFactory;
import org.unitils.util.AnnotationUtils;
import org.unitils.util.ReflectionUtils;


/**
 * Extended {@link Steps}: This class will add the correct steps to the runner.
 *
 * @author Willemijn Wouters
 *
 * @since 1.0.0
 *
 */
public class UnitilsSteps extends Steps {

    /**
     * @param configuration : the configuration of JBehave.
     * @param type : the type of the step.
     * @param stepsFactory : the stepsfactory.
     */
    public UnitilsSteps(Configuration configuration, Class<?> type, InjectableStepsFactory stepsFactory) {
        super(configuration, type, stepsFactory);

    }

    /**
     * Lists all the {@link BeforeOrAfterStep}s of a specific type.
     *
     * @param type : any/normal/example
     * @return {@link List}
     * @see org.jbehave.core.steps.Steps#listBeforeOrAfterScenario(org.jbehave.core.annotations.ScenarioType)
     */
    @Override
    public List<BeforeOrAfterStep> listBeforeOrAfterScenario(ScenarioType type) {
        List<BeforeOrAfterStep> steps = new ArrayList<BeforeOrAfterStep>();
        StepCreator stepCreator = createStepCreator();
        InjectableStepsFactory stepsFactory = getStepsFactory();

        steps.addAll(UnitilsStepsFactory.createBeforeTestClassStep(stepCreator, stepsFactory, type()));
        steps.addAll(UnitilsStepsFactory.createAfterCreateTestObject(stepCreator, stepsFactory, type()));
        steps.addAll(UnitilsStepsFactory.createBeforeTestSetUp(stepCreator, stepsFactory, type()));

        steps.addAll(scenarioStepsHaving(type, Stage.BEFORE, BeforeScenario.class));
        steps.addAll(UnitilsStepsFactory.createBeforeTestMethod(stepCreator, stepsFactory, type()));
        steps.addAll(UnitilsStepsFactory.createAfterTestMethod(stepCreator, stepsFactory, type()));
        steps.addAll(scenarioStepsHaving(type, Stage.AFTER, AfterScenario.class, ANY, SUCCESS, FAILURE));
        steps.addAll(UnitilsStepsFactory.createAfterTestTearDown(stepCreator, stepsFactory, type()));
        return Collections.unmodifiableList(steps);
    }

    /**
     * Creates a list of scenarios by Stage.
     *
     * @param type : the ScenarioType.
     * @param stage : the stage (before/after).
     * @param annotationClass : the annotation
     * @param outcomes : an {@link java.lang.reflect.Array} of outcomes.
     * @return {@link List}
     */
    protected List<BeforeOrAfterStep> scenarioStepsHaving(ScenarioType type, Stage stage, Class<? extends Annotation> annotationClass, Outcome... outcomes) {
        List<BeforeOrAfterStep> steps = new ArrayList<BeforeOrAfterStep>();
        for (Method method : AnnotationUtils.getMethodsAnnotatedWith(type(), annotationClass)) {
            ScenarioType scenarioType = scenarioType(method, annotationClass);
            if (type == scenarioType) {
                if (stage == Stage.BEFORE) {
                    steps.add(createBeforeOrAfterStep(stage, method));
                }
                if (stage == Stage.AFTER) {
                    Outcome scenarioOutcome = scenarioOutcome(method, annotationClass);
                    for (Outcome outcome : outcomes) {
                        if (outcome.equals(scenarioOutcome)) {
                            steps.add(createBeforeOrAfterStep(stage, method, outcome));
                        }
                    }
                }
            }
        }
        return steps;
    }

    /**
     * This methods gets the {@link ScenarioType} out of the {@link BeforeScenario} or {@link AfterScenario}.
     *
     * @param method : the method in the step
     * @param annotationClass : check if this annotation exists on the method
     * @return {@link ScenarioType}
     */
    protected ScenarioType scenarioType(Method method, Class<? extends Annotation> annotationClass) {
        if (annotationClass.isAssignableFrom(BeforeScenario.class)) {
            return ((BeforeScenario) method.getAnnotation(annotationClass)).uponType();
        }
        if (annotationClass.isAssignableFrom(AfterScenario.class)) {
            return ((AfterScenario) method.getAnnotation(annotationClass)).uponType();
        }
        return ScenarioType.NORMAL;
    }

    /**
     * This method creates a basic {@link BeforeOrAfterStep}.
     *
     * @param stage : before/after
     * @param method : the method in the step.
     * @return {@link BeforeOrAfterStep}
     */
    protected BeforeOrAfterStep createBeforeOrAfterStep(Stage stage, Method method) {
        return createBeforeOrAfterStep(stage, method, Outcome.ANY);
    }

    /**
     * This method creates a basic {@link BeforeOrAfterStep}.
     *
     * @param stage : before/after
     * @param method : the method in the step
     * @param outcome : success/failure/any
     * @return {@link BeforeOrAfterStep}
     */
    protected BeforeOrAfterStep createBeforeOrAfterStep(Stage stage, Method method, Outcome outcome) {
        return new BasicUnitilsSteps(stage, method, outcome, createStepCreator());
    }

    /**
     * Get the {@link org.jbehave.core.annotations.AfterScenario.Outcome} out of the {@link AfterScenario}.
     *
     * @param method : the method in the step.
     * @param annotationClass : check if this annotation exists on the method.
     * @return {@link org.jbehave.core.annotations.AfterScenario.Outcome}
     */
    protected Outcome scenarioOutcome(Method method, Class<? extends Annotation> annotationClass) {
        if (annotationClass.isAssignableFrom(AfterScenario.class)) {
            return ((AfterScenario) method.getAnnotation(annotationClass)).uponOutcome();
        }
        return Outcome.ANY;
    }

    /**
     * This method creates a new {@link UnitilsStepCandidate}.
     *
     * @param method : check if this annotation exists on the method.
     * @param stepType : the type of the step
     * @param stepPatternAsString : the pattern of the step
     * @param priority : the priority of the step
     * @param configuration : the JBehave configuration
     * @return {@link StepCandidate}
     */
    protected StepCandidate createCandidate(Method method, StepType stepType, String stepPatternAsString, int priority, Configuration configuration) {
        return new UnitilsStepCandidate(stepPatternAsString, priority, stepType, method, type(), getStepsFactory(), configuration.keywords(), configuration.stepPatternParser(), configuration.parameterConverters(), configuration.parameterControls());
    }

    /**
     * This method creates a new {@link UnitilsStepCreator}.
     *
     * @return {@link StepCreator}
     */
    public StepCreator createStepCreator() {
        return new UnitilsStepCreator(type(), getStepsFactory(), configuration().parameterConverters(), configuration().parameterControls(), null, configuration().stepMonitor());
    }

    /**
     * This is a getter for the stepsFactory.
     *
     * @return {@link InjectableStepsFactory}
     */
    public InjectableStepsFactory getStepsFactory() {
        Field fieldStepsFactory = ReflectionUtils.getFieldWithName(Steps.class, "stepsFactory", false);
        return ReflectionUtils.getFieldValue(this, fieldStepsFactory);
    }
}
