package org.unitils.jbehave.core.parsers;

import static java.util.regex.Pattern.DOTALL;
import static java.util.regex.Pattern.compile;
import static org.apache.commons.lang.StringUtils.removeStart;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jbehave.core.annotations.AfterScenario.Outcome;
import org.jbehave.core.configuration.Keywords;
import org.jbehave.core.i18n.LocalizedKeywords;
import org.jbehave.core.model.Description;
import org.jbehave.core.model.ExamplesTable;
import org.jbehave.core.model.ExamplesTableFactory;
import org.jbehave.core.model.GivenStories;
import org.jbehave.core.model.Lifecycle;
import org.jbehave.core.model.Lifecycle.Steps;
import org.jbehave.core.model.Meta;
import org.jbehave.core.model.Narrative;
import org.jbehave.core.model.Scenario;
import org.jbehave.core.model.Story;
import org.jbehave.core.parsers.RegexStoryParser;


/**
 * extended {@link RegexStoryParser}: the comments shouldn't be added to the description.
 *
 * @author Willemijn Wouters
 *
 * @since 1.0.1
 *
 */
public class RegexStoryParserExtended extends RegexStoryParser {

    protected static final Log LOGGER = LogFactory.getLog(RegexStoryParserExtended.class);

    private static final String NONE = "";

    private Keywords keywords;

    private ExamplesTableFactory tableFactory;

    /**
     *
     */
    public RegexStoryParserExtended() {
        this(new LocalizedKeywords());
    }

    /**
     * @param tableFactory : the factory that creates the {@link ExamplesTable}.
     */
    public RegexStoryParserExtended(ExamplesTableFactory tableFactory) {
        this(new LocalizedKeywords(), tableFactory);
    }

    /**
     * @param keywords : the {@link Keywords} used by JBehave.
     */
    public RegexStoryParserExtended(Keywords keywords) {
        this(keywords, new ExamplesTableFactory());
    }

    /**
     * @param keywords : the {@link Keywords} used by JBehave.
     * @param tableFactory : the factory that creates the {@link ExamplesTable}.
     */
    public RegexStoryParserExtended(Keywords keywords, ExamplesTableFactory tableFactory) {
        super(keywords, tableFactory);
        this.keywords = keywords;
        this.tableFactory = tableFactory;
    }


    /**
     * @param storyAsText : the entire story as text
     * @param storyPath : the path of the story
     * @return {@link Story}
     * @see org.jbehave.core.parsers.RegexStoryParser#parseStory(java.lang.String, java.lang.String)
     */
    @Override
    public Story parseStory(String storyAsText, String storyPath) {
        Description description = parseDescriptionFrom(storyAsText);
        Meta meta = parseStoryMetaFrom(storyAsText);
        Narrative narrative = parseNarrativeFrom(storyAsText);
        GivenStories givenStories = parseGivenStories(storyAsText);
        Lifecycle lifecycle = parseLifecycle(storyAsText);
        List<Scenario> scenarios = parseScenariosFrom(storyAsText);
        Story story = new Story(storyPath, description, meta, narrative, givenStories, lifecycle, scenarios);
        if (storyPath != null) {
            story.namedAs(new File(storyPath).getName());
        }
        return story;
    }

    /**
     * Parse the entire {@link Description} out of the story text.
     *
     * @param storyAsText : the entire story as text.
     * @return {@link Description}
     */
    protected Description parseDescriptionFrom(String storyAsText) {
        Matcher findingDescription = patternToPullDescriptionIntoGroupOne().matcher(storyAsText);
        if (findingDescription.matches()) {
            return new Description(findingDescription.group(1).trim());
        }
        return Description.EMPTY;
    }

    /**
     * This method parses all the meta data from the story text.
     *
     * @param storyAsText : the entire story as text.
     * @return {@link Meta}
     * @see <a href="http://jbehave.org/reference/stable/meta-info.html">meta info</a>
     */
    protected Meta parseStoryMetaFrom(String storyAsText) {
        Matcher findingMeta = patternToPullStoryMetaIntoGroupOne().matcher(preScenarioText(storyAsText));
        if (findingMeta.matches()) {
            String meta = findingMeta.group(1).trim();
            return Meta.createMeta(meta, keywords);
        }
        return Meta.EMPTY;
    }

    /**
     * Get the pre-scenario text out of the story text.
     *
     * @param storyAsText : the entire story as text.
     * @return {@link String}
     */
    protected String preScenarioText(String storyAsText) {
        String[] split = storyAsText.split(keywords.scenario());
        return split.length > 0 ? split[0] : storyAsText;
    }

    /**
     * This method pulls the {@link Narrative} out of the story text.
     *
     * @param storyAsText : the entire story as text.
     * @return {@link Narrative}
     */
    protected Narrative parseNarrativeFrom(String storyAsText) {
        Matcher findingNarrative = patternToPullNarrativeIntoGroupOne().matcher(storyAsText);
        if (findingNarrative.matches()) {
            String narrative = findingNarrative.group(1).trim();
            return createNarrative(narrative);
        }
        return Narrative.EMPTY;
    }

    /**
     * This method pulls the narrative out of the story text.
     *
     * @param narrative : the narrative as plain text.
     * @return {@link Narrative}
     * @see <a href="http://jbehave.org/reference/stable/story-syntax.html">narrative</a>
     */
    protected Narrative createNarrative(String narrative) {
        Matcher findingElements = patternToPullNarrativeElementsIntoGroups().matcher(narrative);
        if (findingElements.matches()) {
            String inOrderTo = findingElements.group(1).trim();
            String asA = findingElements.group(2).trim();
            String iWantTo = findingElements.group(3).trim();
            return new Narrative(inOrderTo, asA, iWantTo);
        }
        Matcher findingAlternativeElements = patternToPullAlternativeNarrativeElementsIntoGroups().matcher(narrative);
        if (findingAlternativeElements.matches()) {
            String asA = findingAlternativeElements.group(1).trim();
            String iWantTo = findingAlternativeElements.group(2).trim();
            String soThat = findingAlternativeElements.group(3).trim();
            return new Narrative("", asA, iWantTo, soThat);
        }
        return Narrative.EMPTY;
    }

    /**
     * Gets all the {@link GivenStories} out of the story text.
     *
     * @param storyAsText : the entire story as text.
     * @return {@link GivenStories}
     * @see <a href="http://jbehave.org/reference/stable/given-stories.html">given stories</a>
     */
    protected GivenStories parseGivenStories(String storyAsText) {
        String scenarioKeyword = keywords.scenario();
        // use text before scenario keyword, if found
        String beforeScenario = "";
        if (StringUtils.contains(storyAsText, scenarioKeyword)) {
            beforeScenario = StringUtils.substringBefore(storyAsText, scenarioKeyword);
        }
        Matcher findingGivenStories = patternToPullStoryGivenStoriesIntoGroupOne().matcher(beforeScenario);
        String givenStories = findingGivenStories.find() ? findingGivenStories.group(1).trim() : NONE;
        return new GivenStories(givenStories);
    }

    /**
     * Parses the {@link Lifecycle} out of the story text.
     *
     * @param storyAsText : the entire story as text.
     * @return {@link Lifecycle}
     */
    protected Lifecycle parseLifecycle(String storyAsText) {
        String scenarioKeyword = keywords.scenario();
        // use text before scenario keyword, if found
        String beforeScenario = "";
        if (StringUtils.contains(storyAsText, scenarioKeyword)) {
            beforeScenario = StringUtils.substringBefore(storyAsText, scenarioKeyword);
        }
        Matcher findingLifecycle = patternToPullLifecycleIntoGroupOne().matcher(beforeScenario);
        String lifecycle = findingLifecycle.find() ? findingLifecycle.group(1).trim() : NONE;
        Matcher findingBeforeAndAfter = compile(".*" + keywords.before() + "(.*)\\s*" + keywords.after() + "(.*)\\s*", DOTALL).matcher(lifecycle);
        if (findingBeforeAndAfter.matches()) {
            String beforeLifecycle = findingBeforeAndAfter.group(1).trim();
            Steps beforeSteps = parseBeforeLifecycle(beforeLifecycle);
            String afterLifecycle = findingBeforeAndAfter.group(2).trim();
            Steps[] afterSteps = parseAfterLifecycle(afterLifecycle);
            return new Lifecycle(beforeSteps, afterSteps);
        }
        Matcher findingBefore = compile(".*" + keywords.before() + "(.*)\\s*", DOTALL).matcher(lifecycle);
        if (findingBefore.matches()) {
            String beforeLifecycle = findingBefore.group(1).trim();
            Steps beforeSteps = parseBeforeLifecycle(beforeLifecycle);
            return new Lifecycle(beforeSteps, new Steps(new ArrayList<String>()));
        }
        Matcher findingAfter = compile(".*" + keywords.after() + "(.*)\\s*", DOTALL).matcher(lifecycle);
        if (findingAfter.matches()) {
            Steps beforeSteps = Steps.EMPTY;
            String afterLifecycle = findingAfter.group(1).trim();
            Steps[] afterSteps = parseAfterLifecycle(afterLifecycle);
            return new Lifecycle(beforeSteps, afterSteps);
        }
        return Lifecycle.EMPTY;
    }

    /**
     * parses the steps before the lifecycle.
     *
     * @param lifecycleAsText : the entire lifecycle as text.
     * @return {@link org.jbehave.core.model.Lifecycle.Steps}
     */
    protected Steps parseBeforeLifecycle(String lifecycleAsText) {
        return new Steps(findSteps(startingWithNL(lifecycleAsText)));
    }

    /**
     * This method parses all the steps after the {@link Keywords#lifecycle()} in the story text.
     *
     * @param lifecycleAsText : the entire lifecycle as text.
     * @return {@link java.lang.reflect.Array}
     */
    protected Steps[] parseAfterLifecycle(String lifecycleAsText) {
        List<Steps> list = new ArrayList<Steps>();
        for (String stepsByOutcome : lifecycleAsText.split(keywords.outcome())) {
            if (stepsByOutcome.trim().isEmpty())
                continue;
            String outcomeAsText = findOutcome(stepsByOutcome);
            List<String> steps = findSteps(startingWithNL(StringUtils.removeStart(stepsByOutcome.trim(), outcomeAsText)));
            list.add(new Steps(parseOutcome(outcomeAsText), steps));
        }
        return list.toArray(new Steps[list.size()]);
    }

    /**
     * This method parses the outcome out of the story text.
     *
     * @param outcomeAsText : the outcome as plain text.
     * @return {@link org.jbehave.core.annotations.AfterScenario.Outcome}
     * @see <a href="http://jbehave.org/reference/stable/story-syntax.html">syntax story</a>
     */
    protected Outcome parseOutcome(String outcomeAsText) {
        if (outcomeAsText.equals(keywords.outcomeSuccess())) {
            return Outcome.SUCCESS;
        } else if (outcomeAsText.equals(keywords.outcomeFailure())) {
            return Outcome.FAILURE;
        }
        return Outcome.ANY;
    }

    /**
     * This method looks for the outcome of a step.
     *
     * @param stepsByOutcome : all the steps by outcome.
     * @return {@link String}
     * @see <a href="http://jbehave.org/reference/stable/story-syntax.html">syntax story</a>
     */
    protected String findOutcome(String stepsByOutcome) {
        Matcher findingOutcome = patternToPullLifecycleOutcomeIntoGroupOne().matcher(stepsByOutcome);
        if (findingOutcome.matches()) {
            return findingOutcome.group(1).trim();
        }
        return keywords.outcomeAny();
    }

    /**
     * This method parses the story text in diffenent {@link Scenario}s.
     *
     * @param storyAsText : the entire story as text.
     * @return {@link List}
     */
    protected List<Scenario> parseScenariosFrom(String storyAsText) {
        List<Scenario> parsed = new ArrayList<Scenario>();
        for (String scenarioAsText : splitScenarios(storyAsText)) {
            parsed.add(parseScenario(scenarioAsText));
        }
        return parsed;
    }

    /**
     * This methods looks for the word {@link Keywords#scenario()} in the story text and splits the story text up in diffent strings (the
     * scenarios).
     *
     * @param storyAsText : the entire story as text.
     * @return {@link List}
     */
    protected List<String> splitScenarios(String storyAsText) {
        List<String> scenarios = new ArrayList<String>();
        String scenarioKeyword = keywords.scenario();

        // use text after scenario keyword, if found
        if (StringUtils.contains(storyAsText, scenarioKeyword)) {
            storyAsText = StringUtils.substringAfter(storyAsText, scenarioKeyword);
        }

        for (String scenarioAsText : storyAsText.split(scenarioKeyword)) {
            if (scenarioAsText.trim().length() > 0) {
                scenarios.add(scenarioKeyword + "\n" + scenarioAsText);
            }
        }

        return scenarios;
    }

    /**
     * This method pulls the entire {@link Scenario} out of the story text.
     *
     * @param scenarioAsText : the entire scenario as plain text.
     * @return {@link Scenario}
     */
    protected Scenario parseScenario(String scenarioAsText) {
        String title = findScenarioTitle(scenarioAsText);
        String scenarioWithoutKeyword = removeStart(scenarioAsText, keywords.scenario()).trim();
        String scenarioWithoutTitle = removeStart(scenarioWithoutKeyword, title);
        scenarioWithoutTitle = startingWithNL(scenarioWithoutTitle);
        Meta meta = findScenarioMeta(scenarioWithoutTitle);
        ExamplesTable examplesTable = findExamplesTable(scenarioWithoutTitle);
        GivenStories givenStories = findScenarioGivenStories(scenarioWithoutTitle);
        if (givenStories.requireParameters()) {
            givenStories.useExamplesTable(examplesTable);
        }
        List<String> steps = findSteps(scenarioWithoutTitle);
        return new Scenario(title, meta, givenStories, examplesTable, steps);
    }

    /**
     * ensure that the text starts on a new line.
     * 
     * @param text : the text that needs to be checked
     * @return {@link String}
     */
    protected String startingWithNL(String text) {
        if (!text.startsWith("\n")) { // always ensure starts with newline
            return "\n" + text;
        }
        return text;
    }

    /**
     * This method looks for the scenario title in the story text.
     *
     * @param scenarioAsText : the entire scenario as plain text.
     * @return {@link String}
     */
    protected String findScenarioTitle(String scenarioAsText) {
        Matcher findingTitle = patternToPullScenarioTitleIntoGroupOne().matcher(scenarioAsText);
        return findingTitle.find() ? findingTitle.group(1).trim() : NONE;
    }

    /**
     * This method looks for the {@link Meta} data in the store text.
     *
     * @param scenarioAsText : the entire scenario as plain text.
     * @return {@link Meta}
     * @see <a href="http://jbehave.org/reference/stable/meta-info.html">meta</a>
     */
    protected Meta findScenarioMeta(String scenarioAsText) {
        Matcher findingMeta = patternToPullScenarioMetaIntoGroupOne().matcher(scenarioAsText);
        if (findingMeta.matches()) {
            String meta = findingMeta.group(1).trim();
            return Meta.createMeta(meta, keywords);
        }
        return Meta.EMPTY;
    }

    /**
     * This method finds the {@link ExamplesTable} out of the story text.
     *
     * @param scenarioAsText : the entire scenario as plain text.
     * @return {@link ExamplesTable}
     * @see <a href="http://jbehave.org/reference/stable/parametrised-scenarios.html">examples table</a>
     */
    protected ExamplesTable findExamplesTable(String scenarioAsText) {
        Matcher findingTable = patternToPullExamplesTableIntoGroupOne().matcher(scenarioAsText);
        String tableInput = findingTable.find() ? findingTable.group(1).trim() : NONE;
        return tableFactory.createExamplesTable(tableInput);
    }

    /**
     * This method finds all the {@link GivenStories} out of the story text.
     *
     * @param scenarioAsText : the entire scenario as plain text.
     * @return {@link GivenStories}
     * @see <a href="http://jbehave.org/reference/stable/given-stories.html">given stories</a>
     */
    protected GivenStories findScenarioGivenStories(String scenarioAsText) {
        Matcher findingGivenStories = patternToPullScenarioGivenStoriesIntoGroupOne().matcher(scenarioAsText);
        String givenStories = findingGivenStories.find() ? findingGivenStories.group(1).trim() : NONE;
        return new GivenStories(givenStories);
    }

    /**
     * This method gets all the different steps out of the story text.
     *
     * @param stepsAsText : all the steps as one plain text.
     * @return {@link List}
     * @see <a href="http://jbehave.org/reference/stable/annotations.html">steps</a>
     */
    protected List<String> findSteps(String stepsAsText) {
        Matcher matcher = patternToPullStepsIntoGroupOne().matcher(stepsAsText);
        List<String> steps = new ArrayList<String>();
        int startAt = 0;
        while (matcher.find(startAt)) {
            steps.add(StringUtils.substringAfter(matcher.group(1), "\n"));
            startAt = matcher.start(4);
        }
        return steps;
    }

    // Regex Patterns
    /**
     * This method creates a regex {@link Pattern} to get the description out of {@link Meta}, {@link Narrative}, {@link Lifecycle},
     * {@link Scenario}s.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullDescriptionIntoGroupOne() {
        String metaOrNarrativeOrLifecycleOrScenario = concatenateWithOr(keywords.meta(), keywords.narrative(), keywords.lifecycle(), keywords.scenario());
        return compile("(.*?)(" + metaOrNarrativeOrLifecycleOrScenario + ").*", DOTALL);
    }

    /**
     * create a regex {@link Pattern} to get the {@link Meta} data out of the story text.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullStoryMetaIntoGroupOne() {
        String narrativeOrGivenStories = concatenateWithOr(keywords.narrative(), keywords.givenStories());
        return compile(".*" + keywords.meta() + "(.*?)\\s*(\\Z|" + narrativeOrGivenStories + ").*", DOTALL);
    }

    /**
     * Creates a regex {@link Pattern} to get all the {@link Narrative}s out of the first group.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullNarrativeIntoGroupOne() {
        String givenStoriesOrLifecycleOrScenario = concatenateWithOr(keywords.givenStories(), keywords.lifecycle(), keywords.scenario());
        return compile(".*" + keywords.narrative() + "(.*?)\\s*(" + givenStoriesOrLifecycleOrScenario + ").*", DOTALL);
    }

    /**
     * Creates a regex pattern to get all the {@link Narrative}s out of the story text.
     *
     * @return {@link Pattern}
     * @see <a href="http://jbehave.org/reference/latest/grammar.html">grammer</a>
     */
    protected Pattern patternToPullNarrativeElementsIntoGroups() {
        return compile(".*" + keywords.inOrderTo() + "(.*)\\s*" + keywords.asA() + "(.*)\\s*" + keywords.iWantTo() + "(.*)", DOTALL);
    }

    /**
     * Creates a regex pattern to get all the alternative {@link Narrative}s out of the story text.
     *
     * @return {@link Pattern}
     * @see <a href="http://jbehave.org/reference/latest/grammar.html">grammer</a>
     */
    protected Pattern patternToPullAlternativeNarrativeElementsIntoGroups() {
        return compile(".*" + keywords.asA() + "(.*)\\s*" + keywords.iWantTo() + "(.*)\\s*" + keywords.soThat() + "(.*)", DOTALL);
    }

    /**
     * This method is used to create the regex pattern to parse the {@link GivenStories} out of the story text.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullStoryGivenStoriesIntoGroupOne() {
        String lifecycleOrScenario = concatenateWithOr(keywords.lifecycle(), keywords.scenario());
        return compile(".*" + keywords.givenStories() + "(.*?)\\s*(\\Z|" + lifecycleOrScenario + ").*", DOTALL);
    }

    /**
     * This method creates the regex expression to pull the {@link Lifecycle} out of the story text.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullLifecycleIntoGroupOne() {
        return compile(".*" + keywords.lifecycle() + "\\s*(.*)", DOTALL);
    }

    /**
     * This method creates the regex pattern to get the lifecycle out of the story text.
     *
     * @return {@link Pattern}
     * @see <a href="http://jbehave.org/reference/latest/story-syntax.html">story-syntax</a>
     */
    protected Pattern patternToPullLifecycleOutcomeIntoGroupOne() {
        String startingWords = concatenateWithOr("\\n", "", keywords.startingWords());
        String outcomes = concatenateWithOr(keywords.outcomeAny(), keywords.outcomeSuccess(), keywords.outcomeFailure());
        return compile("\\s*(" + outcomes + ")\\s*(" + startingWords + ").*", DOTALL);
    }

    /**
     * Creates a regex pattern to get the scenario title out of the story text.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullScenarioTitleIntoGroupOne() {
        String startingWords = concatenateWithOr("\\n", "", keywords.startingWords());
        return compile(keywords.scenario() + "((.)*?)\\s*(" + keywords.meta() + "|" + startingWords + ").*", DOTALL);
    }

    /**
     * Create a regex pattern to get all the {@link Meta} information out of the story text.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullScenarioMetaIntoGroupOne() {
        String startingWords = concatenateWithOr("\\n", "", keywords.startingWords());
        return compile(".*" + keywords.meta() + "(.*?)\\s*(" + keywords.givenStories() + "|" + startingWords + ").*", DOTALL);
    }

    /**
     * Create a regex pattern to get the {@link Keywords#givenStories()} out of the story text.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullScenarioGivenStoriesIntoGroupOne() {
        String startingWords = concatenateWithOr("\\n", "", keywords.startingWords());
        return compile("\\n" + keywords.givenStories() + "((.|\\n)*?)\\s*(" + startingWords + ").*", DOTALL);
    }

    /**
     * We removed the {@link Keywords#ignorable()} keyword out of the list. In the original code JBehave will put all the comments in the
     * {@link Description}. And this is something that we don't want.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullStepsIntoGroupOne() {
        String[] startingWords = keywords.startingWords();
        startingWords = (String[]) ArrayUtils.removeElement(startingWords, keywords.ignorable());
        String initialStartingWords = concatenateWithOr("\\n", "", startingWords);
        String followingStartingWords = concatenateWithOr("\\n", "\\s", startingWords);
        return compile("((" + initialStartingWords + ")\\s(.)*?)\\s*(\\Z|" + followingStartingWords + "|\\n" + keywords.examplesTable() + ")", DOTALL);
    }

    /**
     * create a regex pattern to look for the {@link ExamplesTable}.
     *
     * @return {@link Pattern}
     */
    protected Pattern patternToPullExamplesTableIntoGroupOne() {
        return compile("\\n" + keywords.examplesTable() + "\\s*(.*)", DOTALL);
    }

    /**
     * concatenate the keywords with an | in between, f.e.: keyword1|keyword2|keyword3 .
     *
     * @param keywords : the strings that needs to be concatenated.
     * @return {@link String}
     */
    protected String concatenateWithOr(String... keywords) {
        return concatenateWithOr(null, null, keywords);
    }

    /**
     * concatenate the keywords with an | in between, f.e.: before keyword1 after|before keyword2 after|before keyword3 after.
     *
     * @param beforeKeyword : the keyword that should be appended before each element of the param keywords.
     * @param afterKeyword : the keyword that should be appended after each element of the param keywords.
     * @param keywords : the {@link Keywords} used by JBehave.
     * @return {@link String}
     */
    protected String concatenateWithOr(String beforeKeyword, String afterKeyword, String[] keywords) {
        StringBuilder builder = new StringBuilder();
        String before = beforeKeyword != null ? beforeKeyword : NONE;
        String after = afterKeyword != null ? afterKeyword : NONE;
        for (String keyword : keywords) {
            builder.append(before).append(keyword).append(after).append("|");
        }
        return StringUtils.chomp(builder.toString(), "|"); // chop off the last
                                                           // "|"
    }
}
