/*
 * Copyright (c) Smals
 */
package org.unitils.jbehave.core.reporters;

import java.io.File;

import org.jbehave.core.reporters.NullStoryReporter;
import org.unitils.jbehave.core.reporters.ScreenshotStrategy.StepType;
import org.unitils.selenium.ScreenshotTakingWebDriver;

/**
 * Reporter that will take screenshots at certain moments of the scenario (configurable).
 */
public class SeleniumScreenshotReporter extends NullStoryReporter {

    private SeleniumSteps steps;
    private String directory;
    private String scenarioTitle;
    
    private ScreenshotStrategy strategy;
    
    public SeleniumScreenshotReporter(String directory, SeleniumSteps steps) {
        this(directory, steps, ScreenshotStrategy.FAILING_ONLY);
    }
    
    public SeleniumScreenshotReporter(String directory, SeleniumSteps steps, ScreenshotStrategy strategy) {
        this.steps = steps;
        this.directory = directory;
        this.strategy = strategy;
    }
    
    @Override
    public void beforeScenario(String scenarioTitle) {
        this.scenarioTitle = scenarioTitle;
    }
    
    @Override
    public void beforeStep(String step) {
        takeScreenshot(step, StepType.BEFORE_STEP);
    }
    
    @Override
    public void failed(String step, Throwable cause) {
        takeScreenshot(step, StepType.AFTER_STEP_FAILED);
    }

    private void takeScreenshot(String step, StepType stepType) {
        try {
            ScreenshotTakingWebDriver driver = steps.getScreenshotTakingWebDriver();
            if (driver != null && strategy.shouldTakeScreenshot(stepType)) {
                driver.saveScreenshot(getDirectory(), getFileName(step));
            }
        } catch (Throwable t) {
            // do not fail test because it was not possible to take screenshot
        }
    }
    
    private File getDirectory() {
        switch (strategy) {
            case ALWAYS:
                return new File("./target/" + directory + "/" + scenarioTitle.replaceAll(" ", "_"));
            case FAILING_ONLY:
            default:
                return new File("./target/screenshot/" + directory);
        }
    }
    
    private String getFileName(String step) {
        switch (strategy) {
            case ALWAYS:
                return step.replaceAll(" ", "_");
            case FAILING_ONLY:
            default:
                return scenarioTitle.replaceAll(" ", "_");
        }
    }
}
