package org.unitils.jbehave.core.reporters;

import java.lang.reflect.Field;

import org.jbehave.core.configuration.Keywords;
import org.jbehave.core.failures.UUIDExceptionWrapper;
import org.jbehave.core.reporters.ConsoleOutput;
import org.jbehave.core.reporters.PrintStreamOutput;
import org.jbehave.core.reporters.StackTraceFormatter;
import org.unitils.util.ReflectionUtils;


/**
 * Print the stacktraces directly and do not wait until the end.
 *
 * @author Willemijn Wouters
 *
 * @since 1.0.1
 *
 */
public class UnitilsConsoleOutput extends ConsoleOutput {

    /**
     * @param keywords ; the keywords defined in JBehave.
     */
    public UnitilsConsoleOutput(Keywords keywords) {
        super(keywords);
    }

    /**
     * Print the reason why the step has failed.
     *
     * @param step : the name of the step
     * @param storyFailure : the throwable that is thrown by the step.
     * @see org.jbehave.core.reporters.PrintStreamOutput#failed(java.lang.String, java.lang.Throwable)
     */
    @Override
    public void failed(String step, Throwable storyFailure) {
        // storyFailure be used if a subclass has rewritten the "failed" pattern
        // to have a {3} as WebDriverHtmlOutput (jbehave-web) does.
        if (storyFailure instanceof UUIDExceptionWrapper) {
            String stackTrace = createStackTrace(storyFailure);
            print(format("failed", "{0} ({1})\n({2})\n", step, getKeywords().failed(), ((UUIDExceptionWrapper) storyFailure).getUUID()));
            print(stackTrace);
        } else {
            throw new ClassCastException(storyFailure + " should be an instance of UUIDExceptionWrapper");
        }
    }

    /**
     * This method creates a stacktrace from a {@link Throwable}.
     *
     * @param throwable : the throwable of the failed step
     * @return {@link String}
     */
    protected String createStackTrace(Throwable throwable) {
        return new StackTraceFormatter(compressFailureTrace()).stackTrace(throwable);
    }

    /**
     * Getter for the field keywords.
     *
     * @return {@link Keywords}
     */
    protected Keywords getKeywords() {
        Field field = ReflectionUtils.getFieldWithName(PrintStreamOutput.class, "keywords", false);
        field.setAccessible(true);
        return ReflectionUtils.getFieldValue(this, field);
    }
}
