package org.unitils.jbehave.core.reporters;

import static org.jbehave.core.steps.StepCreator.PARAMETER_TABLE_END;
import static org.jbehave.core.steps.StepCreator.PARAMETER_TABLE_START;
import static org.jbehave.core.steps.StepCreator.PARAMETER_VALUE_END;
import static org.jbehave.core.steps.StepCreator.PARAMETER_VALUE_START;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jbehave.core.configuration.Keywords;
import org.jbehave.core.model.ExamplesTable;
import org.jbehave.core.model.GivenStories;
import org.jbehave.core.model.Lifecycle;
import org.jbehave.core.model.Meta;
import org.jbehave.core.model.Narrative;
import org.jbehave.core.model.OutcomesTable;
import org.jbehave.core.model.Scenario;
import org.jbehave.core.model.Story;
import org.jbehave.core.model.StoryDuration;
import org.jbehave.core.reporters.EscapeMode;
import org.jbehave.core.reporters.StackTraceFormatter;
import org.jbehave.core.reporters.StoryReporter;
import org.jbehave.core.reporters.TemplateProcessor;

import freemarker.ext.beans.BeansWrapper;
import freemarker.template.TemplateHashModel;
import freemarker.template.TemplateModelException;

/**
 * @author Suzan Rogge
 * @since 04/11/2014
 */
public class UnitilsHtmlReporter implements StoryReporter {

    private static final Log LOGGER = LogFactory.getLog(UnitilsHtmlReporter.class);

    private final File file;
    private final Keywords keywords;
    private final TemplateProcessor processor;
    private final String templatePath;
    protected OutputStory outputStory = new OutputStory();
    protected OutputScenario outputScenario = new OutputScenario();
    private OutputStep failedStep;

    public UnitilsHtmlReporter(File file, Keywords keywords) {
        this(file, keywords, "ftl/unitils-html-output.ftl");
    }
    
    /**
     * Constructor.
     *
     * @param file : the report file
     * @param keywords : the keywords used by JBehave
     * @param templatePath : the path of the template
     */
    public UnitilsHtmlReporter(File file, Keywords keywords, String templatePath) {
        this.file = file;
        this.keywords = keywords;
        this.templatePath = templatePath;
        this.processor = new UnitilsFreemarkerProcessor();
    }

    /**
     * The story is not allowed.
     *
     * @param story : the story
     * @param filter : filter
     * @see org.jbehave.core.reporters.StoryReporter#storyNotAllowed(org.jbehave.core.model.Story, java.lang.String)
     */
    @Override
    public void storyNotAllowed(Story story, String filter) {
        this.outputStory.notAllowedBy = filter;
    }

    /**
     * Do all the things that need to be done before the story is reported.
     *
     * @param story : the story
     * @param givenStory : is it the given story?
     * @see org.jbehave.core.reporters.StoryReporter#beforeStory(org.jbehave.core.model.Story, boolean)
     */
    @Override
    public void beforeStory(Story story, boolean givenStory) {
        if (!givenStory) {
            this.outputStory = new OutputStory();
            this.outputStory.description = story.getDescription().asString();
            this.outputStory.path = story.getPath();
        }
        if (!story.getMeta().isEmpty()) {
            this.outputStory.meta = new OutputMeta(story.getMeta());
        }
    }

    /**
     * set the narrative.
     *
     * @param narrative : the narrative that needs to be set
     *
     * @see org.jbehave.core.reporters.StoryReporter#narrative(org.jbehave.core.model.Narrative)
     */
    @Override
    public void narrative(Narrative narrative) {
        if (!narrative.isEmpty()) {
            this.outputStory.narrative = new OutputNarrative(narrative);
        }
    }

    /**
     * setter for the lifecycle.
     *
     * @param lifecycle : the lifecycle that needs to be set
     *
     * @see org.jbehave.core.reporters.StoryReporter#lifecyle(org.jbehave.core.model.Lifecycle)
     */
    @Override
    public void lifecyle(Lifecycle lifecycle) {
        if(!lifecycle.isEmpty()){
            this.outputStory.lifecycle = new OutputLifecycle(lifecycle);
        }
    }

    /**
     * sets the scenario as not allowed.
     *
     * @param scenario : the scenario
     * @param filter : the filter for the outputscenario
     *
     * @see org.jbehave.core.reporters.StoryReporter#scenarioNotAllowed(org.jbehave.core.model.Scenario, java.lang.String)
     */
    @Override
    public void scenarioNotAllowed(Scenario scenario, String filter) {
        this.outputScenario.notAllowedBy = filter;
    }

    /**
     * sets the title of the outputscenario.
     *
     * @param title : the title of the outputscenario.
     *
     * @see org.jbehave.core.reporters.StoryReporter#beforeScenario(java.lang.String)
     */
    @Override
    public void beforeScenario(String title) {
        if (this.outputScenario.currentExample == null) {
            this.outputScenario = new OutputScenario();
        }
        this.outputScenario.title = title;
    }

    /**
     * @param step : the name of the step
     * @see org.jbehave.core.reporters.StoryReporter#beforeStep(java.lang.String)
     */
    @Override
    public void beforeStep(String step) {
        //do nothing at the moment
    }

    /**
     * Creates a new successful outputstep and adds it to the scenario.
     *
     * @param step : the name of the step
     *
     * @see org.jbehave.core.reporters.StoryReporter#successful(java.lang.String)
     */
    @Override
    public void successful(String step) {
        this.outputScenario.addStep(new OutputStep(step, "successful"));
    }

    /**
     * adds a new ignorable outputstep to the outputscenario.
     *
     * @param step : the name of the step.
     *
     * @see org.jbehave.core.reporters.StoryReporter#ignorable(java.lang.String)
     */
    @Override
    public void ignorable(String step) {
        this.outputScenario.addStep(new OutputStep(step, "ignorable"));
    }

    /**
     * adds a new pending step to the outputscenario.
     *
     * @param step : the name of the step
     * @see org.jbehave.core.reporters.StoryReporter#pending(java.lang.String)
     */
    @Override
    public void pending(String step) {
        this.outputScenario.addStep(new OutputStep(step, "pending"));
    }

    /**
     * adds a new not performed outputstep to the outputscenario.
     *
     * @param step : the name of the step.
     *
     * @see org.jbehave.core.reporters.StoryReporter#notPerformed(java.lang.String)
     */
    @Override
    public void notPerformed(String step) {
        this.outputScenario.addStep(new OutputStep(step, "notPerformed"));
    }

    /**
     * adds a new failed outputstep to the outputscenario.
     *
     * @param step : the name of the step.
     * @param storyFailure : the story failure.
     * @see org.jbehave.core.reporters.StoryReporter#failed(java.lang.String, java.lang.Throwable)
     */
    @Override
    public void failed(String step, Throwable storyFailure) {
        this.failedStep = new OutputStep(step, "failed");
        failedStep.failure = storyFailure;
        this.outputScenario.addStep(failedStep);
    }

    /**
     * adds a new failed outputstep to the outputscenario.
     *
     * @param step : the name of the step.
     * @param table : the {@link OutcomesTable}.
     *
     * @see org.jbehave.core.reporters.StoryReporter#failedOutcomes(java.lang.String, org.jbehave.core.model.OutcomesTable)
     */
    @Override
    public void failedOutcomes(String step, OutcomesTable table) {
        failed(step, table.failureCause());
        this.failedStep.outcomes = table;
    }

    /**
     * setter givenStories.
     *
     * @param givenStories : the stories defined in the test.
     * @see org.jbehave.core.reporters.StoryReporter#givenStories(org.jbehave.core.model.GivenStories)
     */
    @Override
    public void givenStories(GivenStories givenStories) {
        if (!givenStories.getStories().isEmpty()) {
            this.outputScenario.givenStories = givenStories;
        }
    }

    /**
     * Creates a new GivenStories for each path.
     *
     * @param storyPaths : list with all the stories defined in the test.
     * @see org.jbehave.core.reporters.StoryReporter#givenStories(java.util.List)
     */
    @Override
    public void givenStories(List<String> storyPaths) {
        givenStories(new GivenStories(StringUtils.join(storyPaths, ",")));
    }

    /**
     * Sets the {@link Meta} value.
     *
     * @param meta : The meta data defined in the story.
     *
     * @see org.jbehave.core.reporters.StoryReporter#scenarioMeta(org.jbehave.core.model.Meta)
     */
    @Override
    public void scenarioMeta(Meta meta) {
        if (!meta.isEmpty()) {
            this.outputScenario.meta = new OutputMeta(meta);
        }
    }

    /**
     * Sets the steps and the examplestable.
     *
     * @param steps : a list with steps.
     * @param table : the examplestable.
     * @see org.jbehave.core.reporters.StoryReporter#beforeExamples(java.util.List, org.jbehave.core.model.ExamplesTable)
     */
    @Override
    public void beforeExamples(List<String> steps, ExamplesTable table) {
        this.outputScenario.examplesSteps = steps;
        this.outputScenario.examplesTable = table;
    }

    /**
     * sets the example & currentexample.
     *
     * @param parameters : examples
     * @see org.jbehave.core.reporters.StoryReporter#example(java.util.Map)
     */
    @Override
    public void example(Map<String, String> parameters) {
        this.outputScenario.examples.add(parameters);
        this.outputScenario.currentExample = parameters;
    }

    /**
     * sets the currentExample to null.
     *
     * @see org.jbehave.core.reporters.StoryReporter#afterExamples()
     */
    @Override
    public void afterExamples() {
        this.outputScenario.currentExample = null;
    }

    /**
     * @see org.jbehave.core.reporters.StoryReporter#dryRun()
     */
    @Override
    public void dryRun() {
        //do nothing for now
    }

    /**
     * @see org.jbehave.core.reporters.StoryReporter#afterScenario()
     */
    @Override
    public void afterScenario() {
        if (this.outputScenario.currentExample == null) {
            this.outputStory.scenarios.add(outputScenario);
        }
    }
    /**
     * setter pendingMethods
     *
     * @param methods : list of methods.
     *
     * @see org.jbehave.core.reporters.StoryReporter#pendingMethods(java.util.List)
     */
    @Override
    public void pendingMethods(List<String> methods) {
        this.outputStory.pendingMethods = methods;
    }

    /**
     * adds a new {@link OutputRestart} step.
     *
     * @param step : the name of the step.
     * @param cause : the reason why the step failed.
     * @see org.jbehave.core.reporters.StoryReporter#restarted(java.lang.String, java.lang.Throwable)
     */
    @Override
    public void restarted(String step, Throwable cause) {
        this.outputScenario.addStep(new OutputRestart(step, cause.getMessage()));
    }

    /**
     * sets the story as cancelled.
     *
     * @param story : the story defined in your test.
     * @param storyDuration : the duration of the story.
     * @see org.jbehave.core.reporters.StoryReporter#storyCancelled(org.jbehave.core.model.Story, org.jbehave.core.model.StoryDuration)
     */
    @Override
    public void storyCancelled(Story story, StoryDuration storyDuration) {
        this.outputStory.cancelled = true;
        this.outputStory.storyDuration = storyDuration;
    }

    /**
     * Do some rapportation after the story.
     *
     * @param givenStory : is this the given story?
     *
     * @see org.jbehave.core.reporters.StoryReporter#afterStory(boolean)
     */
    @Override
    public void afterStory(boolean givenStory) {
        if (!givenStory) {
            Map<String, Object> model = newDataModel();
            model.put("story", outputStory);
            model.put("keywords", new OutputKeywords(keywords));

            TemplateHashModel enumModels = getEnumModels();
            TemplateHashModel escapeEnums;
            try {
                String escapeModeEnum = EscapeMode.class.getCanonicalName();
                escapeEnums = (TemplateHashModel) enumModels.get(escapeModeEnum);
                model.put("EscapeMode", escapeEnums);
            } catch (TemplateModelException e) {
                throw new IllegalArgumentException(e);
            }

            write(file, templatePath, model);
        }
    }

    /**
     * Get the enumModels from the {@link BeansWrapper}.
     *
     * @return {@link TemplateHashModel}
     */
    protected TemplateHashModel getEnumModels() {
        BeansWrapper wrapper = BeansWrapper.getDefaultInstance();
        return wrapper.getEnumModels();
    }
    /**
     * Write the output to the file.
     *
     * @param file : the file where everything should be written to.
     * @param resource : the path of the resource
     * @param dataModel : the datamodel
     */
    protected void write(File file, String resource, Map<String, Object> dataModel) {
        try {
            file.getParentFile().mkdirs();
            Writer writer = createFileWriter(file);
            processor.process(resource, dataModel, writer);
            writer.close();
        } catch (Exception e) {
            LOGGER.error("error occurred while creating JBehave report: " + e.getMessage());
        }
    }

    /**
     * Create a new data model.
     *
     * @return {@link Map}
     */
    protected Map<String, Object> newDataModel() {
        return new HashMap<String, Object>();
    }

    /**
     * Create a {@link FileWriter} for a specific  file.
     *
     * @param file : the file for the filewriter.
     * @return {@link FileWriter}
     * @throws IOException
     */
    protected FileWriter createFileWriter(File file) throws IOException {
        return new FileWriter(file);
    }


    /**
     * outputScenario setter.
     *
     * @param outputScenario the outputScenario to set
     */
    protected void setOutputScenario(OutputScenario outputScenario) {
        this.outputScenario = outputScenario;
    }

    /**
     * getter outputStory.
     *
     * @return {@link OutputStory}
     */
    protected OutputStory getOutputStory() {
        return outputStory;
    }


    /**
     * getter outputScenario
     *
     * @return {@link OutputScenario}
     */
    protected OutputScenario getOutputScenario() {
        return outputScenario;
    }

    /**
     * OutputKeywords.
     */
    public static class OutputKeywords {

        private final Keywords keywords;

        /**
         * Constructor {@link OutputKeywords}
         *
         * @param keywords : the keywords defined by JBehave
         */
        public OutputKeywords(Keywords keywords) {
            this.keywords = keywords;
        }
        
        /**
         * getter keyword lifecycle.
         *
         * @return {@link String}
         */
        public String getLifecycle(){
            return keywords.lifecycle();
        }

        /**
         * getter keyword before.
         *
         * @return {@link String}
         */
        public String getBefore(){
            return keywords.before();
        }

        /**
         * getter keyword after.
         *
         * @return {@link String}
         */
        public String getAfter(){
            return keywords.after();
        }

        /**
         * getter keyword meta.
         *
         * @return {@link String}
         */
        public String getMeta() {
            return keywords.meta();
        }

        /**
         * getter keyword metaProperty.
         *
         * @return {@link String}
         */
        public String getMetaProperty() {
            return keywords.metaProperty();
        }

        /**
         * getter keyword narrative.
         *
         * @return {@link String}
         */
        public String getNarrative() {
            return keywords.narrative();
        }

        /**
         * getter keyword inOrderTo.
         *
         * @return {@link String}
         */
        public String getInOrderTo() {
            return keywords.inOrderTo();
        }

        /**
         * getter keyword asA.
         *
         * @return {@link String}
         */
        public String getAsA() {
            return keywords.asA();
        }

        /**
         * getter keyword iWantTo.
         *
         * @return {@link String}
         */
        public String getiWantTo() {
            return keywords.iWantTo();
        }

        /**
         * getter keyword soThat.
         *
         * @return {@link String}
         */
        public String getSoThat() {
            return keywords.soThat();
        }

        /**
         * getter keyword scenario.
         *
         * @return {@link String}
         */
        public String getScenario() {
            return keywords.scenario();
        }

        /**
         * getter keywords givenStories.
         *
         * @return {@link String}
         */
        public String getGivenStories() {
            return keywords.givenStories();
        }

        /**
         * getter keyword examplesTable.
         *
         * @return {@link String}
         */
        public String getExamplesTable() {
            return keywords.examplesTable();
        }

        /**
         * getter keyword examplesTableRow.
         *
         * @return {@link String}
         */
        public String getExamplesTableRow() {
            return keywords.examplesTableRow();
        }

        /**
         * getter keyword examplesTableHeaderSeparator.
         *
         * @return {@link String}
         */
        public String getExamplesTableHeaderSeparator() {
            return keywords.examplesTableHeaderSeparator();
        }

        /**
         * getter keyword examplesTableValueSeparator.
         *
         * @return {@link String}
         */
        public String getExamplesTableValueSeparator() {
            return keywords.examplesTableValueSeparator();
        }

        /**
         * getter keyword examplesTableIgnorableSeparator.
         *
         * @return {@link String}
         */
        public String getExamplesTableIgnorableSeparator() {
            return keywords.examplesTableIgnorableSeparator();
        }

        /**
         * getter keyword given.
         *
         * @return {@link String}
         */
        public String getGiven() {
            return keywords.given();
        }

        /**
         * getter keyword when.
         *
         * @return {@link String}
         */
        public String getWhen() {
            return keywords.when();
        }

        /**
         * getter keyword then.
         *
         * @return {@link String}
         */
        public String getThen() {
            return keywords.then();
        }

        /**
         * getter keyword and.
         *
         * @return {@link String}
         */
        public String getAnd() {
            return keywords.and();
        }

        /**
         * getter keyword ignorable.
         *
         * @return {@link String}
         */
        public String getIgnorable() {
            return keywords.ignorable();
        }

        /**
         * getter keyword pending.
         *
         * @return {@link String}
         */
        public String getPending() {
            return keywords.pending();
        }

        /**
         * getter keyword notPerformed.
         *
         * @return {@link String}
         */
        public String getNotPerformed() {
            return keywords.notPerformed();
        }

        /**
         * getter keyword failed.
         *
         * @return {@link String}
         */
        public String getFailed() {
            return keywords.failed();
        }

        /**
         * getter keyword dryRun.
         *
         * @return {@link String}
         */
        public String getDryRun() {
            return keywords.dryRun();
        }

        /**
         * getter keyword storyCancelled.
         *
         * @return {@link String}
         */
        public String getStoryCancelled() {
            return keywords.storyCancelled();
        }

        /**
         * getter keyword duration.
         *
         * @return {@link String}
         */
        public String getDuration() {
            return keywords.duration();
        }

        /**
         * getter keyword yes.
         *
         * @return {@link String}
         */
        public String getYes() {
            return keywords.yes();
        }

        /**
         * getter keyword no.
         *
         * @return {@link String}
         */
        public String getNo() {
            return keywords.no();
        }
    }
    /**
     * OutputStory.
     */
    public static class OutputStory {
        private String description;
        private String path;
        private OutputMeta meta;
        private OutputNarrative narrative;
        private OutputLifecycle lifecycle;
        private String notAllowedBy;
        private List<String> pendingMethods;
        private List<OutputScenario> scenarios = new ArrayList<OutputScenario>();
        private boolean cancelled;
        private StoryDuration storyDuration;

        /**
         * getter description.
         *
         * @return {@link String}
         */
        public String getDescription() {
            return description;
        }

        /**
         * getter path.
         *
         * @return {@link String}
         */
        public String getPath() {
            return path;
        }

        /**
         * getter meta.
         *
         * @return {@link OutputMeta}
         */
        public OutputMeta getMeta() {
            return meta;
        }

        /**
         * getter narrative.
         *
         * @return {@link OutputNarrative}
         */
        public OutputNarrative getNarrative() {
            return narrative;
        }

        /**
         * getter lifecycle.
         *
         * @return {@link OutputLifecycle}
         */
        public OutputLifecycle getLifecycle() {
            return lifecycle;
        }

        /**
         * getter notAllowedBy.
         *
         * @return {@link String}
         */
        public String getNotAllowedBy() {
            return notAllowedBy;
        }

        /**
         * getter pendingMethods.
         *
         * @return {@link List}
         */
        public List<String> getPendingMethods() {
            return pendingMethods;
        }

        /**
         * getter scenarios.
         *
         * @return {@link List}
         */
        public List<OutputScenario> getScenarios() {
            return scenarios;
        }

        /**
         * getter cancelled.
         *
         * @return {@link Boolean}
         */
        public boolean isCancelled() {
            return cancelled;
        }

        /**
         * getter storyDuration.
         *
         * @return {@link StoryDuration}
         */
        public StoryDuration getStoryDuration() {
            return storyDuration;
        }
    }

    /**
     * OutputMeta.
     */
    public static class OutputMeta {

        private final Meta meta;

        /**
         * Constructor.
         *
         * @param meta : {@link Meta} data defined in story.
         */
        public OutputMeta(Meta meta) {
            this.meta = meta;
        }

        /**
         * Gets all the properties out of the {@link Meta} and puts them in a {@link Map}
         *
         * @return {@link Map}
         */
        public Map<String, String> getProperties() {
            Map<String, String> properties = new HashMap<String, String>();
            for (String name : meta.getPropertyNames()) {
                properties.put(name, meta.getProperty(name));
            }
            return properties;
        }

    }

    /**
     * OutputNarrative.
     */
    public static class OutputNarrative {
        private final Narrative narrative;

        /**
         * Constructor.
         *
         * @param narrative : the narrative
         */
        public OutputNarrative(Narrative narrative) {
            this.narrative = narrative;
        }

        /**
         * This method gets the inOrderTo variable out of the {@link Narrative}.
         *
         * @return {@link String}
         */
        public String getInOrderTo() {
            return narrative.inOrderTo();
        }

        /**
         * This method gets the asA variable out of the {@link Narrative}.
         *
         * @return {@link String}
         */
        public String getAsA() {
            return narrative.asA();
        }

        /**
         * This method gets the iWantTo variable out of the {@link Narrative}.
         *
         * @return {@link String}
         */
        public String getiWantTo() {
            return narrative.iWantTo();
        }

        /**
         * This method gets the soThat variable out of the {@link Narrative}.
         *
         * @return {@link String}
         */
        public String getSoThat(){
            return narrative.soThat();
        }

        /**
         * This method gets the isAlternative boolean out of the {@link Narrative}.
         *
         * @return {@link Boolean}
         */
        public boolean isAlternative(){
            return narrative.isAlternative();
        }

    }

    /**
     * OutputLifecycle.
     */
    public static class OutputLifecycle {
        private final Lifecycle lifecycle;

        /**
         * Constructor.
         *
         * @param lifecycle : the lifecycle.
         */
        public OutputLifecycle(Lifecycle lifecycle) {
            this.lifecycle = lifecycle;
        }

        /**
         * This method gets the beforeSteps out of the {@link Lifecycle}.
         *
         * @return {@link List}
         */
        public List<String> getBeforeSteps(){
            return lifecycle.getBeforeSteps();
        }

        /**
         * This method gets the aftersteps out of the {@link Lifecycle}.
         *
         * @return {@link List}
         */
        public List<String> getAfterSteps(){
            return lifecycle.getAfterSteps();
        }

    }

    /**
     * Outputscenario.
     */
    public static class OutputScenario {
        private String title;
        private List<OutputStep> steps = new ArrayList<OutputStep>();
        private OutputMeta meta;
        private GivenStories givenStories;
        private String notAllowedBy;
        private List<String> examplesSteps;
        private ExamplesTable examplesTable;
        private Map<String, String> currentExample;
        private List<Map<String, String>> examples = new ArrayList<Map<String, String>>();
        private Map<Map<String, String>, List<OutputStep>> stepsByExample = new HashMap<Map<String, String>, List<OutputStep>>();

        /**
         * getter title.
         *
         * @return {@link String}
         */
        public String getTitle() {
            return title;
        }

        /**
         * Adds the outputstep to the steps (if there is no exampesTable) or to the current example steps.
         *
         * @param outputStep : the outputStep
         */
        public void addStep(OutputStep outputStep) {
            if (examplesTable == null) {
                steps.add(outputStep);
            } else {
                List<OutputStep> currentExampleSteps = stepsByExample.get(currentExample);
                if (currentExampleSteps == null) {
                    currentExampleSteps = new ArrayList<OutputStep>();
                    stepsByExample.put(currentExample, currentExampleSteps);
                }
                currentExampleSteps.add(outputStep);
            }
        }

        /**
         * getter steps
         *
         * @return {@link List}
         */
        public List<OutputStep> getSteps() {
            return steps;
        }

        /**
         * looks if there is already an {@link OutputStep} for this example. If there isn't an {@link OutputStep} for this example than the
         * method will return an empty {@link ArrayList}.
         *
         * @param example : a {@link Map} with examples.
         * @return {@link List}
         */
        public List<OutputStep> getStepsByExample(Map<String, String> example) {
            List<OutputStep> steps = stepsByExample.get(example);
            if (steps == null) {
                return new ArrayList<OutputStep>();
            }
            return steps;
        }

        /**
         * create outcome
         *
         * @return {@link String}
         */
        public String getOutcome() {
            if (notAllowedBy != null) {
                return "skipped";
            }
            for (OutputStep step : steps) {
                String outcome = step.getOutcome();
                if (!outcome.equals("successful") && !outcome.equals("ignorable")) {
                    return outcome;
                }
            }

            return "successful";
        }


        /**
         * @param examplesTable the examplesTable to set
         */
        protected void setExamplesTable(ExamplesTable examplesTable) {
            this.examplesTable = examplesTable;
        }

        /**
         * @param currentExample the currentExample to set
         */
        protected void setCurrentExample(Map<String, String> currentExample) {
            this.currentExample = currentExample;
        }

        /**
         * @param steps the steps to set
         */
        protected void setSteps(List<OutputStep> steps) {
            this.steps = steps;
        }

        /**
         * @param stepsByExample the stepsByExample to set
         */
        protected void setStepsByExample(Map<Map<String, String>, List<OutputStep>> stepsByExample) {
            this.stepsByExample = stepsByExample;
        }

        /**
         * getter meta
         *
         * @return {@link OutputMeta}
         */
        public OutputMeta getMeta() {
            return meta;
        }

        /**
         * getter givenStories.
         *
         * @return {@link GivenStories}
         */
        public GivenStories getGivenStories() {
            return givenStories;
        }

        /**
         * getter notAllowedBy.
         *
         * @return {@link String}
         */
        public String getNotAllowedBy() {
            return notAllowedBy;
        }

        /**
         * getter examplesSteps.
         *
         * @return {@link List}
         */
        public List<String> getExamplesSteps() {
            return examplesSteps;
        }

        /**
         * getter examplesTable.
         *
         * @return {@link ExamplesTable}
         */
        public ExamplesTable getExamplesTable() {
            return examplesTable;
        }

        /**
         * getter examples.
         *
         * @return {@link List}
         */
        public List<Map<String, String>> getExamples() {
            return examples;
        }
    }

    /**
     * OutputRestart.
     */
    public static class OutputRestart extends OutputStep {

        /**
         * Constructor.
         *
         * @param step : the name of the step.
         * @param outcome : the outcome of the step.
         */
        public OutputRestart(String step, String outcome) {
            super(step, outcome);
        }

    }
    /**
     * OutputStep.
     */
    public static class OutputStep {
        private final String step;
        private final String outcome;
        private Throwable failure;
        private OutcomesTable outcomes;
        private List<OutputParameter> parameters;
        private String stepPattern;
        private String tableAsString;
        private ExamplesTable table;

        /**
         * Constructor.
         *
         * @param step : the name of the step
         * @param outcome : the outcome of the step.
         */
        public OutputStep(String step, String outcome) {
            this.step = step;
            this.outcome = outcome;
            parseTableAsString();
            parseParameters();
            createStepPattern();
        }

        /**
         * getter step
         *
         * @return {@link String}
         */
        public String getStep() {
            return step;
        }

        /**
         * getter steppattern.
         *
         * @return {@link String}
         */
        public String getStepPattern() {
            return stepPattern;
        }

        /**
         * return parameters.
         *
         * @return {@link List}
         */
        public List<OutputParameter> getParameters() {
            return parameters;
        }

        /**
         * getter outcome.
         *
         * @return {@link String}
         */
        public String getOutcome() {
            return outcome;
        }

        /**
         * getter {@link Throwable}
         *
         * @return {@link Throwable}
         */
        public Throwable getFailure() {
            return failure;
        }

        /**
         * getter {@link String}
         *
         * @return {@link String}
         */
        public String getFailureCause() {
            if (failure != null) {
                return new StackTraceFormatter(true).stackTrace(failure);
            }
            return "";
        }


        /**
         * @param failure the failure to set
         */
        void setFailure(Throwable failure) {
            this.failure = failure;
        }

        /**
         * getter {@link ExamplesTable}.
         *
         * @return {@link ExamplesTable}
         */
        public ExamplesTable getTable() {
            return table;
        }

        /**
         * getter outcomes.
         *
         * @return {@link OutcomesTable}
         */
        public OutcomesTable getOutcomes() {
            return outcomes;
        }

        /**
         * getter tableAsString
         *
         * @return {@link String}
         */
        public String getTableAsString() {
            return tableAsString;
        }

        /**
         * Get the reason why the step failed.
         *
         * @return {@link String}
         */
        public String getOutcomesFailureCause() {
            if (outcomes.failureCause() != null) {
                return new StackTraceFormatter(true).stackTrace(outcomes.failureCause());
            }
            return "";
        }

        /**
         * Format the pattern
         *
         * @param parameterPattern : the pattern.
         * @return {@link String}
         *
         * @deprecated: formatting without escaping doesn't make sense unless we do a ftl text output format
         */
        @Deprecated
        public String getFormattedStep(String parameterPattern) {
            return getFormattedStep(EscapeMode.NONE, parameterPattern);
        }

        /**
         * Format the step.
         *
         * @param outputFormat : html/xml/none.
         * @param parameterPattern : the pattern of the parameter.
         * @return {@link String}
         */
        public String getFormattedStep(EscapeMode outputFormat, String parameterPattern) {
            // note that escaping the stepPattern string only works
            // because placeholders for parameters do not contain
            // special chars (the placeholder is {0} etc)
            String escapedStep = escapeString(outputFormat, stepPattern);
            if (!parameters.isEmpty()) {
                try {
                    return MessageFormat.format(escapedStep, formatParameters(outputFormat, parameterPattern));
                } catch (RuntimeException e) {
                    throw new StepFormattingFailed(stepPattern, parameterPattern, parameters, e);
                }
            }
            return escapedStep;
        }

        /**
         * Escapes all the special characters.
         *
         * @param outputFormat : This can be html, xml or none.
         * @param string : The string where the special characters should be escaped.
         * @return {@link String}
         */
        protected String escapeString(EscapeMode outputFormat, String string) {
            if(outputFormat==EscapeMode.HTML) {
                return StringEscapeUtils.escapeHtml(string);
            } else if(outputFormat==EscapeMode.XML) {
                return StringEscapeUtils.escapeXml(string);
            } else {
                return string;
            }
        }

        /**
         * The parameters should be formatted by the parameterPattern.
         *
         * @param outputFormat : html/xml/none.
         * @param parameterPattern : the pattern of the parameter.
         * @return {@link java.lang.reflect.Array}
         */
        protected Object[] formatParameters(EscapeMode outputFormat, String parameterPattern) {
            Object[] arguments = new Object[parameters.size()];
            for (int a = 0; a < parameters.size(); a++) {
                arguments[a] = MessageFormat.format(parameterPattern, escapeString(outputFormat, parameters.get(a).getValue()));
            }
            return arguments;
        }

        /**
         * sets the parameter field.
         */
        private void parseParameters() {
            // first, look for parameterized scenarios
            parameters = findParameters(PARAMETER_VALUE_START + PARAMETER_VALUE_START, PARAMETER_VALUE_END + PARAMETER_VALUE_END);
            // second, look for normal scenarios
            if (parameters.isEmpty()) {
                parameters = findParameters(PARAMETER_VALUE_START, PARAMETER_VALUE_END);
            }
        }

        /**
         * This method searches for parameter in the step and creates a {@link List} of {@link OutputParameter}s.
         *
         * @param start : start of the pattern
         * @param end : end of the pattern
         * @return {@link List}
         */
        protected List<OutputParameter> findParameters(String start, String end) {
            List<OutputParameter> parameters = new ArrayList<OutputParameter>();
            Matcher matcher = Pattern.compile("(" + start + ".*?" + end + ")(\\W|\\Z)", Pattern.DOTALL).matcher(step);
            while (matcher.find()) {
                parameters.add(new OutputParameter(step, matcher.start(), matcher.end()));
            }
            return parameters;
        }

        /**
         * Creates a new {@link ExamplesTable}.
         */
        protected void parseTableAsString() {
            if (step.contains(PARAMETER_TABLE_START) && step.contains(PARAMETER_TABLE_END)) {
                tableAsString = StringUtils.substringBetween(step, PARAMETER_TABLE_START, PARAMETER_TABLE_END);
                table = new ExamplesTable(tableAsString);
            }
        }

        /**
         * sets the stepPattern.
         */
        protected void createStepPattern() {
            this.stepPattern = step;
            if (tableAsString != null) {
                this.stepPattern = StringUtils.replaceOnce(stepPattern, PARAMETER_TABLE_START + tableAsString + PARAMETER_TABLE_END, "");
            }
            for (int count = 0; count < parameters.size(); count++) {
                String value = parameters.get(count).toString();
                this.stepPattern = stepPattern.replace(value, "{" + count + "}");
            }
        }

        @SuppressWarnings("serial")
        /**
         * This should be used when there is something wrong with the formatting.
         */
        public static class StepFormattingFailed extends RuntimeException {

            /**
             * Constructor.
             * 
             * @param stepPattern : step pattern
             * @param parameterPattern : parameter pattern
             * @param parameters : list of {@link OutputParameter}
             * @param cause : the cause why the test is failing.
             */
            public StepFormattingFailed(String stepPattern, String parameterPattern, List<OutputParameter> parameters, RuntimeException cause) {
                super("Failed to format step '" + stepPattern + "' with parameter pattern '" + parameterPattern + "' and parameters: " + parameters, cause);
            }

        }

    }

    /**
     * OutputParameter.
     */
    public static class OutputParameter {
        private final String parameter;

        /**
         * Constructor.
         *
         * @param pattern : the parameter pattern
         * @param start : start
         * @param end : end
         */
        public OutputParameter(String pattern, int start, int end) {
            this.parameter = pattern.substring(start, end).trim();
        }

        /**
         * Creates the outputparameter value.
         *
         * @return {@link String}
         */
        public String getValue() {
            String value = StringUtils.remove(parameter, PARAMETER_VALUE_START);
            value = StringUtils.remove(value, PARAMETER_VALUE_END);
            return value;
        }

        @Override
        public String toString() {
            return parameter;
        }
    }

}
