package org.unitils.jbehave.core.stepcreator;

import static org.jbehave.core.steps.AbstractStepResult.failed;
import static org.jbehave.core.steps.AbstractStepResult.skipped;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;

import org.jbehave.core.annotations.AfterScenario.Outcome;
import org.jbehave.core.annotations.Named;
import org.jbehave.core.configuration.Keywords;
import org.jbehave.core.failures.BeforeOrAfterFailed;
import org.jbehave.core.failures.UUIDExceptionWrapper;
import org.jbehave.core.model.Meta;
import org.jbehave.core.parsers.StepMatcher;
import org.jbehave.core.steps.InjectableStepsFactory;
import org.jbehave.core.steps.ParameterControls;
import org.jbehave.core.steps.ParameterConverters;
import org.jbehave.core.steps.Step;
import org.jbehave.core.steps.StepCreator;
import org.jbehave.core.steps.StepMonitor;
import org.jbehave.core.steps.StepResult;
import org.unitils.jbehave.core.steps.IUnitilsStep;
import org.unitils.util.ReflectionUtils;

import com.thoughtworks.paranamer.Paranamer;


/**
 * Unitils Steps are invoked a little bit different, so the {@link UnitilsStepCreator} is a little bit modified.
 *
 * @author Willemijn Wouters
 *
 * @since 1.0.0
 *
 */
public class UnitilsStepCreator extends StepCreator {

    /**
     * Constructor.
     *
     * @param stepsType : the type of the step.
     * @param stepsFactory : an {@link InjectableStepsFactory}
     * @param parameterConverters : a {@link ParameterConverters}
     * @param parameterControls : a {@link ParameterControls}
     * @param stepMatcher : a {@link StepMatcher}
     * @param stepMonitor : a {@link StepMonitor}
     */
    public UnitilsStepCreator(Class<?> stepsType, InjectableStepsFactory stepsFactory, ParameterConverters parameterConverters, ParameterControls parameterControls, StepMatcher stepMatcher, StepMonitor stepMonitor) {
        super(stepsType, stepsFactory, parameterConverters, parameterControls, stepMatcher, stepMonitor);
    }

    /**
     * This method creates a new {@link StepCreatorBeforeOrAfterStep}.
     *
     * @param meta : the {@link Meta} data defined in the story.
     * @param method : the method
     * @return {@link Step}
     * @see org.jbehave.core.steps.StepCreator#createBeforeOrAfterStep(java.lang.reflect.Method, org.jbehave.core.model.Meta)
     */
    @Override
    public Step createBeforeOrAfterStep(Method method, Meta meta) {
        return new StepCreatorBeforeOrAfterStep(method, meta);
    }

    /**
     * Creates a new {@link Step} depending on the {@link org.jbehave.core.model.Outcome}.
     *
     * @param method : the method
     * @param outcome : any/success/failure
     * @param storyAndScenarioMeta : The {@link Meta} defined in the story and scenario.
     * @return {@link Step}
     * @see org.jbehave.core.steps.StepCreator#createAfterStepUponOutcome(java.lang.reflect.Method,
     *      org.jbehave.core.annotations.AfterScenario.Outcome, org.jbehave.core.model.Meta)
     */
    @Override
    public Step createAfterStepUponOutcome(Method method, Outcome outcome, Meta storyAndScenarioMeta) {
        switch (outcome) {
            case ANY:
            default:
                return new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta);
            case SUCCESS:
                return new UnitilsUponSuccessStep(method, storyAndScenarioMeta);
            case FAILURE:
                return new UnitilsFailureStep(method, storyAndScenarioMeta);
        }
    }

    /**
     * Creates a new {@link Step}
     *
     * @see org.jbehave.core.steps.StepCreator#createAfterStepUponOutcome(java.lang.reflect.Method,
     *      org.jbehave.core.annotations.AfterScenario.Outcome, org.jbehave.core.model.Meta)
     * @param method : the method
     * @param outcome : any/success/failure
     * @param storyAndScenarioMeta : The {@link Meta} defined in the story and scenario
     * @param step : a {@link org.jbehave.core.steps.BeforeOrAfterStep}
     * @return {@link Step}
     */
    public Step createAfterStepUponOutcome(Method method, Outcome outcome, Meta storyAndScenarioMeta, org.jbehave.core.steps.BeforeOrAfterStep step) {
        switch (outcome) {
            case ANY:
            default:
                return new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta, step);
            case SUCCESS:
                return new UnitilsUponSuccessStep(method, storyAndScenarioMeta, step);
            case FAILURE:
                return new UnitilsFailureStep(method, storyAndScenarioMeta, step);
        }
    }

    /**
     *
     * In {@link org.jbehave.core.annotations.AfterScenario.Outcome} you can define when this scenario step should be invoked. This method
     * will only run when there is a previous step that fails.
     *
     */
    protected class UnitilsFailureStep extends UponFailureStep {

        private StepCreatorBeforeOrAfterStep beforeOrAfterStep;

        /**
         * @param method : the method
         * @param storyAndScenarioMeta : The {@link Meta} defined in the story and scenario
         * @param beforeOrAfterStep : a {@link org.jbehave.core.steps.BeforeOrAfterStep}
         */
        public UnitilsFailureStep(Method method, Meta storyAndScenarioMeta, org.jbehave.core.steps.BeforeOrAfterStep beforeOrAfterStep) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta, beforeOrAfterStep);
        }

        /**
         * @param method : the method
         * @param storyAndScenarioMeta : The {@link Meta} defined in the story and scenario
         */
        public UnitilsFailureStep(Method method, Meta storyAndScenarioMeta) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta);
        }

        @Override
        public StepResult doNotPerform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return beforeOrAfterStep.perform(storyFailureIfItHappened);
        }

        @Override
        public StepResult perform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return skipped();
        }

        @Override
        public String asString(Keywords keywords) {
            return beforeOrAfterStep.asString(keywords);
        }

    }

    /**
     *
     * In {@link org.jbehave.core.annotations.AfterScenario.Outcome} you can define when this scenario step should be invoked. This method
     * will only run when there is a previous step that succeeds.
     *
     */
    protected class UnitilsUponSuccessStep extends UponSuccessStep {

        private StepCreatorBeforeOrAfterStep beforeOrAfterStep;

        /**
         * @param method : the method
         * @param storyAndScenarioMeta : The {@link Meta} defined in the story and scenario
         * @param step : a {@link org.jbehave.core.steps.BeforeOrAfterStep}
         */
        public UnitilsUponSuccessStep(Method method, Meta storyAndScenarioMeta, org.jbehave.core.steps.BeforeOrAfterStep step) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta, step);
        }

        /**
         * @param method : the method
         * @param storyAndScenarioMeta : The {@link Meta} defined in the story and scenario
         */
        public UnitilsUponSuccessStep(Method method, Meta storyAndScenarioMeta) {
            super(method, storyAndScenarioMeta);
            this.beforeOrAfterStep = new StepCreatorBeforeOrAfterStep(method, storyAndScenarioMeta);
        }

        @Override
        public StepResult doNotPerform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return skipped();
        }

        @Override
        public StepResult perform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return beforeOrAfterStep.perform(storyFailureIfItHappened);
        }

        @Override
        public String asString(Keywords keywords) {
            return beforeOrAfterStep.asString(keywords);
        }
    }

    /**
     * Extract parameter names using {@link Named}-annotated parameters
     *
     * @param method the Method with {@link Named}-annotated parameters
     * @return An array of annotated parameter names, which <b>may</b> include <code>null</code> values for parameters that are not
     *         annotated
     */
    protected String[] annotatedParameterNames(Method method) {
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();
        String[] names = new String[parameterAnnotations.length];
        for (int i = 0; i < parameterAnnotations.length; i++) {
            for (Annotation annotation : parameterAnnotations[i]) {
                names[i] = annotationName(annotation);
            }
        }
        return names;
    }

    /**
     * Returns either the value of the annotation, either {@link Named} or "javax.inject.Named".
     *
     * @param annotation the Annotation
     * @return The annotated value or <code>null</code> if no annotation is found
     */
    protected String annotationName(Annotation annotation) {
        if (annotation.annotationType().isAssignableFrom(Named.class)) {
            return ((Named) annotation).value();
        } else if ("javax.inject.Named".equals(annotation.annotationType().getName())) {
            return Jsr330Helper.getNamedValue(annotation);
        } else {
            return null;
        }
    }

    /**
     * This is a different class, because the @Inject jar may not be in the classpath.
     */
    public static class Jsr330Helper {

        /**
         * Get the value from an annotation.
         *
         * @param annotation : an {@link Annotation}
         * @return {@link String}
         */
        public static String getNamedValue(Annotation annotation) {
            return ((Named) annotation).value();
        }

    }
    /**
     *
     * This class is just like a BeforeOrAFterStep in the {@link StepCreator}. But before
     *
     */
    public class StepCreatorBeforeOrAfterStep extends AbstractStep {


        private final Method method;

        private final Meta meta;

        private IUnitilsStep unitilsStep;

        /**
         * Constructor.
         *
         * @param method : the method
         * @param meta : The {@link Meta} used by the story.
         */
        public StepCreatorBeforeOrAfterStep(Method method, Meta meta) {
            this.method = method;
            this.meta = meta;
        }

        /**
         * Constructor.
         *
         * @param method : the method
         * @param meta : The {@link Meta} used by the story.
         * @param step : a {@link org.jbehave.core.steps.BeforeOrAfterStep}
         */
        public StepCreatorBeforeOrAfterStep(Method method, Meta meta, org.jbehave.core.steps.BeforeOrAfterStep step) {
            this(method, meta);
            if (step instanceof IUnitilsStep) {
                this.unitilsStep = (IUnitilsStep) step;
            }
        }

        /**
         * invoke the method from the step.
         *
         * @param storyFailureIfItHappened : the failure
         * @return {@link StepResult}
         * @see org.jbehave.core.steps.Step#perform(org.jbehave.core.failures.UUIDExceptionWrapper)
         */
        @Override
        public StepResult perform(UUIDExceptionWrapper storyFailureIfItHappened) {
            ParameterConverters paramConvertersWithExceptionInjector = paramConvertersWithExceptionInjector(storyFailureIfItHappened);
            MethodInvoker methodInvoker = new MethodInvoker(method, paramConvertersWithExceptionInjector, getParanamer(), meta);

            try {
                if (unitilsStep != null) {
                    unitilsStep.invokeUnitilsMethod();
                } else {
                    methodInvoker.invoke();
                }
            } catch (InvocationTargetException e) {
                return failed(method, new UUIDExceptionWrapper(new BeforeOrAfterFailed(method, e.getCause())));
            } catch (Throwable t) {
                return failed(method, new UUIDExceptionWrapper(new BeforeOrAfterFailed(method, t)));
            }

            return skipped();
        }

        /**
         * Add a new {@link org.jbehave.core.steps.ParameterConverters.ParameterConverter}
         *
         * @param storyFailureIfItHappened : the failure
         * @return {@link ParameterConverters}
         */
        protected ParameterConverters paramConvertersWithExceptionInjector(UUIDExceptionWrapper storyFailureIfItHappened) {
            return getParameterConverters().newInstanceAdding(new UUIDExceptionWrapperInjector(storyFailureIfItHappened));
        }

        /**
         * Do not perform {@link UUIDExceptionWrapper}
         *
         * @param storyFailureIfItHappened : the failure
         * @return {@link StepResult}
         * @see org.jbehave.core.steps.Step#doNotPerform(org.jbehave.core.failures.UUIDExceptionWrapper)
         */
        @Override
        public StepResult doNotPerform(UUIDExceptionWrapper storyFailureIfItHappened) {
            return perform(storyFailureIfItHappened);
        }

        /**
         *
         * {@link org.jbehave.core.steps.ParameterConverters.ParameterConverter} for UUIDExceptions.
         *
         */
        public class UUIDExceptionWrapperInjector implements ParameterConverters.ParameterConverter {

            private final UUIDExceptionWrapper storyFailureIfItHappened;

            /**
             * Constructor.
             *
             * @param storyFailureIfItHappened : the failure
             */
            public UUIDExceptionWrapperInjector(UUIDExceptionWrapper storyFailureIfItHappened) {
                this.storyFailureIfItHappened = storyFailureIfItHappened;
            }

            /**
             * Checks if the parameter type equals {@link UUIDExceptionWrapper}.
             * 
             * @param type : a {@link Type}
             * @return {@link Boolean}
             * @see org.jbehave.core.steps.ParameterConverters.ParameterConverter#accept(java.lang.reflect.Type)
             */
            @Override
            public boolean accept(Type type) {
                return UUIDExceptionWrapper.class == type;
            }

            @Override
            public Object convertValue(String value, Type type) {
                return storyFailureIfItHappened;
            }
        }


        /**
         * getter meta.
         *
         * @return {@link Meta}
         */
        public Meta getMeta() {
            return meta;
        }


        /**
         * getter method.
         *
         * @return {@link Method}
         */
        public Method getMethod() {
            return method;
        }

    }
    /**
     *
     * invokes methods (@see MethodInvoker from {@link StepCreator} (private class))
     *
     */
    protected class MethodInvoker {

        private final Method method;

        private final ParameterConverters parameterConverters;

        private final Paranamer paranamer;

        private final Meta meta;

        private int methodArity;

        /**
         * Constructor.
         *
         * @param method : the {@link Method}
         * @param parameterConverters : a {@link ParameterConverters}
         * @param paranamer : a {@link Paranamer}
         * @param meta : the {@link Meta} defined in the story.
         */
        public MethodInvoker(Method method, ParameterConverters parameterConverters, Paranamer paranamer, Meta meta) {
            this.method = method;
            this.parameterConverters = parameterConverters;
            this.paranamer = paranamer;
            this.meta = meta;
            this.methodArity = method.getParameterTypes().length;
        }

        /**
         * invoke the method field.
         *
         * @throws InvocationTargetException
         * @throws IllegalAccessException
         */
        public void invoke() throws InvocationTargetException, IllegalAccessException {
            method.invoke(stepsInstance(), parameterValuesFrom(meta));
        }

        /**
         * All the parameters of a method.
         *
         * @return array of {@link Parameter}
         */
        protected Parameter[] methodParameters() {
            Parameter[] parameters = new Parameter[methodArity];
            String[] annotationNamedParameters = annotatedParameterNames(method);
            String[] parameterNames = paranamer.lookupParameterNames(method, false);
            Class<?>[] parameterTypes = method.getParameterTypes();

            for (int paramPosition = 0; paramPosition < methodArity; paramPosition++) {
                String paramName = parameterNameFor(paramPosition, annotationNamedParameters, parameterNames);
                parameters[paramPosition] = new Parameter(paramPosition, parameterTypes[paramPosition], paramName);
            }

            return parameters;
        }

        /**
         * Get the name of a specific parameter.
         *
         * @param paramPosition : the position of the wanted parameter.
         * @param annotationNamedParameters : parameters from an annotation.
         * @param parameterNames : all the parameters.
         * @return {@link String}
         */
        protected String parameterNameFor(int paramPosition, String[] annotationNamedParameters, String[] parameterNames) {
            String nameFromAnnotation = nameIfValidPositionInArray(annotationNamedParameters, paramPosition);
            String parameterName = nameIfValidPositionInArray(parameterNames, paramPosition);
            if (nameFromAnnotation != null) {
                return nameFromAnnotation;
            } else if (parameterName != null) {
                return parameterName;
            }
            return null;
        }

        /**
         * Get the name of a specific parameter on a specified position.
         *
         * @param paramNames : all the parameters.
         * @param paramPosition : the position of the wanted parameter.
         * @return {@link String}
         */
        protected String nameIfValidPositionInArray(String[] paramNames, int paramPosition) {
            return paramPosition < paramNames.length ? paramNames[paramPosition] : null;
        }

        /**
         * Get the parameter values from the {@link Meta} data.
         *
         * @param meta : the {@link Meta} defined in the story.
         * @return array of {@link Object}s
         */
        protected Object[] parameterValuesFrom(Meta meta) {
            Object[] values = new Object[methodArity];
            for (Parameter parameter : methodParameters()) {
                values[parameter.position] = parameterConverters.convert(parameter.valueFrom(meta), parameter.type);
            }
            return values;
        }

        /**
         * Represents a parameter in a method.
         */
        protected class Parameter {

            private final int position;

            private final Class<?> type;

            private final String name;

            /**
             * Constructor.
             *
             * @param position : the position the list of the parameters
             * @param type : the class type of the parameter
             * @param name : the name of the parameter
             */
            public Parameter(int position, Class<?> type, String name) {
                this.position = position;
                this.type = type;
                this.name = name;
            }

            /**
             * Get the parameter out of the {@link Meta}.
             *
             * @param meta : the {@link Meta} defined in the story.
             * @return {@link String}
             */
            public String valueFrom(Meta meta) {
                if (name == null) {
                    return null;
                }
                return meta.getProperty(name);
            }
        }
    }
    /**
     * Parameter: name of the parameter + is there an annotation.
     */
    public static class ParameterName {

        private String name;

        private boolean annotated;

        /**
         *
         * @param name : the name of the parameter.
         * @param annotated : is the parameter annotated?
         */
        protected ParameterName(String name, boolean annotated) {
            this.name = name;
            this.annotated = annotated;
        }


        /**
         * getter name
         *
         * @return {@link String}
         */
        public String getName() {
            return name;
        }


        /**
         * getter isAnnotated.
         *
         * @return {@link Boolean}
         */
        public boolean isAnnotated() {
            return annotated;
        }
    }


    /**
     * getter for field parameterConverters.
     *
     * @return {@link ParameterConverters}
     */
    protected ParameterConverters getParameterConverters() {
        Field field = ReflectionUtils.getFieldWithName(StepCreator.class, "parameterConverters", false);
        return ReflectionUtils.getFieldValue(this, field);
    }

    /**
     * getter for field paranamer.
     *
     * @return {@link Paranamer}
     */
    protected Paranamer getParanamer() {
        Field fieldParanamer = ReflectionUtils.getFieldWithName(StepCreator.class, "paranamer", false);
        return ReflectionUtils.getFieldValue(this, fieldParanamer);
    }
}
