package org.unitils.jbehave.core.steps;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jbehave.core.annotations.AfterScenario.Outcome;
import org.jbehave.core.model.Meta;
import org.jbehave.core.steps.BeforeOrAfterStep;
import org.jbehave.core.steps.InjectableStepsFactory;
import org.jbehave.core.steps.Step;
import org.jbehave.core.steps.StepCollector.Stage;
import org.jbehave.core.steps.StepCreator;
import org.unitils.core.TestListener;
import org.unitils.core.Unitils;
import org.unitils.jbehave.core.stepcreator.UnitilsStepCreator;


/**
 * All Unitils steps need a few basics methods and this abstract class provides those methods.
 *
 * @author Willemijn Wouters
 *
 * @since 1.0.0
 *
 */
public abstract class AbstractUnitilsSteps extends BeforeOrAfterStep {

    private static final Log LOGGER = LogFactory.getLog(AbstractUnitilsSteps.class);

    private InjectableStepsFactory stepsFactory;

    private Class<?> clzzStep;

    private StepCreator stepcreator;

    private Method testMethod;

    private Outcome outcome;

    /**
     * Constructor.
     *
     * @param stage : before/after
     * @param method : the {@link Method} of the step.
     * @param testMethod : a {@link Method}
     * @param outcome : any/success/failure
     * @param stepCreator : a {@link StepCreator}
     * @param injectableStepsFactory : an {@link InjectableStepsFactory}.
     * @param type : the type of the step.
     */
    public AbstractUnitilsSteps(Stage stage, Method method, Method testMethod, Outcome outcome, StepCreator stepCreator, InjectableStepsFactory injectableStepsFactory, Class<?> type) {
        super(stage, method, outcome, stepCreator);
        this.stepsFactory = injectableStepsFactory;
        this.clzzStep = type;
        this.stepcreator = stepCreator;
        this.outcome = outcome;
        this.testMethod = testMethod;
    }


    /**
     * Constructor.
     *
     * @param stage : before/after
     * @param method : the {@link Method} of the step.
     * @param testMethod : a {@link Method}
     * @param outcome : any/success/failure
     * @param stepCreator : a {@link StepCreator}
     */
    public AbstractUnitilsSteps(Stage stage, Method method, Method testMethod, Outcome outcome, StepCreator stepCreator) {
        super(stage, method, outcome, stepCreator);
        this.stepcreator = stepCreator;
        this.clzzStep = method.getDeclaringClass();
        this.testMethod = testMethod;
        this.outcome = outcome;
    }


    /**
     * The {@link Unitils} testlistener
     *
     * @return {@link TestListener}
     */
    public TestListener getTestlistener() {
        return Unitils.getInstance().getTestListener();
    }

    /**
     * This method gets the testobject from the stepsfactory.
     *
     * @return {@link Object}
     */
    protected Object getTestObject() {
        if (stepsFactory != null) {
            return getStepsFactory().createInstanceOfType(clzzStep);
        }
        return null;
    }

    /**
     * create a new {@link org.unitils.jbehave.core.stepcreator.UnitilsStepCreator.StepCreatorBeforeOrAfterStep}. Returns null if the
     * {@link StepCreator} isn't of type {@link UnitilsStepCreator}.
     *
     * @param meta : the {@link Meta} data defined in the story.
     * @return {@link Step}
     *
     * @see org.jbehave.core.steps.BeforeOrAfterStep#createStepWith(org.jbehave.core.model.Meta)
     */
    @Override
    public Step createStepWith(Meta meta) {
        if (stepcreator instanceof UnitilsStepCreator) {
            UnitilsStepCreator creator = (UnitilsStepCreator) stepcreator;
            return creator.new StepCreatorBeforeOrAfterStep(getMethod(), meta, this);
        }
        return null;
    }

    /**
     * Create a new {@link Step} with the {@link Meta} data from the story.
     *
     * @param storyAndScenarioMeta : the {@link Meta} data defined in the story of in the scenario.
     * @return {@link Step}
     *
     * @see org.jbehave.core.steps.BeforeOrAfterStep#createStepUponOutcome(org.jbehave.core.model.Meta)
     */
    @Override
    public Step createStepUponOutcome(Meta storyAndScenarioMeta) {
        if (stepcreator instanceof UnitilsStepCreator) {
            UnitilsStepCreator creator = (UnitilsStepCreator) stepcreator;
            return creator.createAfterStepUponOutcome(getMethod(), outcome, storyAndScenarioMeta, this);
        } else {
            return stepcreator.createAfterStepUponOutcome(getMethod(), outcome, storyAndScenarioMeta);
        }

    }

    /**
     * getter stepsFactory.
     *
     * @return the stepsFactory
     */
    public InjectableStepsFactory getStepsFactory() {
        return stepsFactory;
    }

    /**
     * getter clzzStep.
     *
     * @return the clzzStep
     */
    public Class<?> getClzzStep() {
        return clzzStep;
    }

    /**
     * Invoke a method from the the Unitils {@link TestListener}.
     *
     * @throws IllegalAccessException
     * @throws IllegalArgumentException
     * @throws InvocationTargetException
     * @see org.unitils.jbehave.core.steps.IUnitilsStep#invokeUnitilsMethod()
     */
    public void invokeUnitilsMethod() throws IllegalAccessException, IllegalArgumentException, InvocationTargetException {
        LOGGER.debug(getMethod().getName());
        getMethod().invoke(getTestlistener(), getParametersMethod());

    }

    /**
     * get the parameters from the method.
     *
     * @return {@link java.lang.reflect.Array}
     */
    protected abstract Object[] getParametersMethod();


    /**
     * getter testMethod.
     *
     * @return the testMethod
     */
    public Method getTestMethod() {
        return testMethod;
    }


    /**
     * setter testMethod.
     *
     * @param testMethod the testMethod to set
     */
    public void setTestMethod(Method testMethod) {
        this.testMethod = testMethod;
    }
}
