package org.unitils.jodatime;

import java.lang.reflect.Method;
import java.util.Properties;

import org.joda.time.DateTime;
import org.joda.time.DateTimeUtils;
import org.joda.time.Duration;
import org.joda.time.Period;
import org.joda.time.format.DateTimeFormat;
import org.unitils.core.Module;
import org.unitils.core.TestListener;
import org.unitils.jodatime.annotation.FixedDateTime;
import org.unitils.jodatime.annotation.OffsetDateTime;



/**
 * Module to run test with a provided datetime.
 * 
 * Possible uses:
 * <ul>
 * <li>1. time can be frozen to a specific datetime using the {@link FixedDateTime} annotation on test class or method.</li>
 * <li>2. time can be an offset of the current datetime using the {@link OffsetDateTime} annotation on test class or method.</li>
 * </ul>
 * 
 * When an annotation is used at class level it will be used by default for all its test methods. For a specific test method
 * 
 * @author Christophe De Blende
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * 
 * @since 1.0.0 
 * 
 */
public class JodaTimeModule implements Module {

    public void init(Properties configuration) {
        //do nothing
    }

    public void afterInit() {
        //do nothing
    }

    public TestListener getTestListener() {
        return new DateTimeModuleListener();
    }

    protected static class DateTimeModuleListener extends TestListener {

        private long fixedMillis;

        private long offsetMillis;

        @Override
        public void afterTestTearDown(Object testObject, Method testMethod) {
            DateTimeUtils.setCurrentMillisSystem();
        }

        @Override
        public void beforeTestClass(Class<?> testClass) {
            // the module is only instantiated once. So when the beforeTestClass lifecycle callback occurs make sure the members are reset.
            fixedMillis = 0L;
            offsetMillis = 0L;

            if (testClass.isAnnotationPresent(FixedDateTime.class)) {
                FixedDateTime fixedDateTime = testClass.getAnnotation(FixedDateTime.class);
                fixedMillis = calculateFixedMillis(fixedDateTime);
                DateTimeUtils.setCurrentMillisFixed(fixedMillis);


            } else if (testClass.isAnnotationPresent(OffsetDateTime.class)) {
                OffsetDateTime offsetDateTime = testClass.getAnnotation(OffsetDateTime.class);
                offsetMillis = calculateOffsetMillis(offsetDateTime);
                DateTimeUtils.setCurrentMillisOffset(offsetMillis);

            } else {
                DateTimeUtils.setCurrentMillisSystem();
            }
        }

        @Override
        public void beforeTestSetUp(Object testObject, Method testMethod) {
            if (testMethod.isAnnotationPresent(FixedDateTime.class)) {
                FixedDateTime fixedDateTime = testMethod.getAnnotation(FixedDateTime.class);
                DateTimeUtils.setCurrentMillisFixed(calculateFixedMillis(fixedDateTime));

            } else if (testMethod.isAnnotationPresent(OffsetDateTime.class)) {
                OffsetDateTime offsetDateTime = testMethod.getAnnotation(OffsetDateTime.class);
                DateTimeUtils.setCurrentMillisOffset(calculateOffsetMillis(offsetDateTime));

            } else if (fixedMillis != 0L) {
                // if the method doesn't have an annotation for fixeddatetime, but there is one at class level we use the calculated value
                // that was defined at class level.
                DateTimeUtils.setCurrentMillisFixed(fixedMillis);

            } else if (offsetMillis != 0L) {
                // if the method doesn't have an annotation for offsetdatetime, but there is one at class level we use the calculated value
                // that was defined at class level.
                DateTimeUtils.setCurrentMillisOffset(offsetMillis);

            } else {
                // if there is no annotation no where we use the real current time.
                DateTimeUtils.setCurrentMillisSystem();
            }
        }

        private long calculateFixedMillis(FixedDateTime fixedDateTime) {
            if (fixedDateTime.datetime() == null || fixedDateTime.datetime().length() == 0) {
                // default use the current time as a fixed datetime.
                return new DateTime().getMillis();
            }

            DateTime dateTime = DateTimeFormat.forPattern(fixedDateTime.pattern()).parseDateTime(fixedDateTime.datetime());
            return dateTime.getMillis();
        }

        private long calculateOffsetMillis(OffsetDateTime offsetDateTime) {
            Period period = new Period(offsetDateTime.years(), offsetDateTime.months(), offsetDateTime.weeks(), offsetDateTime.days(),
                offsetDateTime.hours(), offsetDateTime.minutes(), offsetDateTime.seconds(), offsetDateTime.millis());


            DateTime currentDate = new DateTime();
            DateTime offisetDate = currentDate.plus(period);

            Duration duration = new Duration(currentDate, offisetDate);
            return duration.getMillis();
        }
    }

}
