/*
 * Copyright 2012,  Unitils.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.unitils.mail;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.unitils.core.Module;
import org.unitils.core.TestListener;
import org.unitils.core.UnitilsException;
import org.unitils.mail.annotation.TestSmtpServer;
import org.unitils.mail.impl.SmtpServerImpl;
import org.unitils.util.AnnotationUtils;
import org.unitils.util.PropertyUtils;
import org.unitils.util.ReflectionUtils;





/**
 * 
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * 
 */
public class MailModule implements Module {



	private static Log LOGGER = LogFactory.getLog(MailModule.class);

    /**
     * property in unitils.properties to set the mail port
     */
    public static final String SMTP_DEFAULT_PORT = "org.unitils.mail.port";

    /**
     * default port where Wiser should start
     */
    private int port;


    /**
     * This method will do the initialization.
     * 
     * This method checks if the user gives a smtp port otherwise the default smtp port (25) will be used.
     * 
     * 
     * @see org.unitils.core.Module#init(java.util.Properties)
     */
    @Override
    public void init(Properties configuration) {
        port = PropertyUtils.getInt(SMTP_DEFAULT_PORT, 25, configuration);
    }


    @Override
	public void afterInit() {
		
	}

    /**
     * This will initialize a new SMTP server and the method will put it in the correct field (SimpleSmtpServer) of the testedObject.
     * 
     * @param testObject of type {@link Object}
     */
    public void initSmtpServer(Object testObject) {
        Set<Field> fields = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), TestSmtpServer.class);

        if (fields.size() > 1) {
            throw new UnitilsException("Multiple SmtpServer declarations found. Module support only one. ");
        }
        if (fields.size() == 1) {
            Field field = fields.iterator().next();
            ReflectionUtils.setFieldValue(testObject, field, new SmtpServerImpl(port));
        }
        
    }


    /**
     * @param testObject
     */
    public void stopSmtpServer(Object testObject) {
        Set<Field> fields = AnnotationUtils.getFieldsAnnotatedWith(testObject.getClass(), TestSmtpServer.class);

        for (Field field : fields) {
            SmtpServer server = ReflectionUtils.getFieldValue(testObject, field);
            server.stop();
            if (server.isRunning()) {
                LOGGER.warn("smtp server was not shut down correctly, port " + port + " could still be open. ");
            }
        }

    }

    /**
     * @see org.unitils.core.Module#getTestListener()
     */
    @Override
    public TestListener getTestListener() {
        return new MailListener();
    }

    /**
     * 
     * Maillistener class
     * 
     * @author thomas de rycke
     * 
     * @since
     * 
     */
    private class MailListener extends TestListener {

        @Override
        public void beforeTestSetUp(Object testObject, Method testMethod) {
            initSmtpServer(testObject);

        }

        @Override
        public void afterTestMethod(Object testObject, Method testMethod, Throwable testThrowable) {
            stopSmtpServer(testObject);
        }
    }

}
