/*
 * Copyright 2012,  Unitils.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.unitils.mail.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.internet.MimeMessage;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.subethamail.wiser.WiserMessage;
import org.unitils.mail.SmtpMessage;


/**
 * Wrapper class for the SMTP message class of Wiser.
 * 
 * @author Jeroen Horemans
 * @author Thomas De Rycke
 * @author Willemijn Wouters
 * 
 */
public class SmtpMessageImpl implements SmtpMessage {
    private String from;

    private String subject;

    private String body;

    private List<String> to;

    private List<String> cc;

    private List<String> bcc;

    private List<byte[]> attachments;


    /**
     * @param from
     * @param subject
     * @param body
     * @param to
     * @throws IOException 
     */
    public SmtpMessageImpl(String from, String subject, String body, String to) throws IOException {
        this(from, subject, body, to, new File[0]);
    }

    /**
     * @param from
     * @param subject
     * @param body
     * @param to
     * @param attachments 
     * @throws IOException 
     */
    public SmtpMessageImpl(String from, String subject, String body, String to, File... attachments) throws IOException {
        this(from, subject, body, Arrays.asList(to), new ArrayList<String>(), attachments);
    }
    /**
     * @param from
     * @param subject
     * @param body
     * @param to
     * @param attachments
     * @throws IOException
     */
    public SmtpMessageImpl(String from, String subject, String body, List<String> to, List<String> cc, File... attachments) throws IOException {
        this(from, subject, body, to, cc, new ArrayList<String>(), attachments);
    }

    /**
     * Constructor of SmtpMessage.
     * 
     * @param from
     * @param subject
     * @param body
     * @param to
     * @param cc 
     * @param bcc
     * @param attachments
     * @throws IOException
     */
    public SmtpMessageImpl(String from, String subject, String body, List<String> to, List<String> cc, List<String> bcc, File...attachments) throws IOException {
        this.from = from;
        this.subject = subject;
        this.body = body;
        if (to != null) {
            this.to = to;
        } else {
            this.to = new ArrayList<String>();
        }
        if (cc != null) {
            this.cc = cc;
        } else {
            this.cc = new ArrayList<String>();
        }
        if (bcc != null) {
            this.bcc = bcc;
        } else {
            this.bcc = new ArrayList<String>();
        }

        if (attachments == null) {
            this.attachments = new ArrayList<byte[]>();
        } else {
            this.attachments = new ArrayList<byte[]>();
            for (File file : attachments) {
                byte[] readFileToByteArray = FileUtils.readFileToByteArray(file);
                this.attachments.add(readFileToByteArray);
            }
        }
    }

    /**
     * Constructor of SmtpMessage.
     * 
     * @param message of type {@link SmtpMessage}
     * @throws MessagingException
     * @throws IOException
     */
    public SmtpMessageImpl(WiserMessage message) throws IOException, MessagingException {
        MimeMessage mimeMessage = message.getMimeMessage();
        this.from = message.getEnvelopeSender();
        this.subject = mimeMessage.getSubject();
        this.body = mimeMessage.getDescription();

        this.to = new ArrayList<String>();
        if (mimeMessage.getRecipients(RecipientType.TO) != null) {

            for (Address address : mimeMessage.getRecipients(RecipientType.TO)) {
                this.to.add(address.toString());
            }
        }

        this.cc = new ArrayList<String>();
        if (mimeMessage.getRecipients(RecipientType.CC) != null) {

            for (Address address : mimeMessage.getRecipients(RecipientType.CC)) {
                this.cc.add(address.toString());
            }
        }

        this.bcc = new ArrayList<String>();
        if (mimeMessage.getRecipients(RecipientType.BCC) != null) {

            for (Address address : mimeMessage.getRecipients(RecipientType.BCC)) {
                this.bcc.add(address.toString());
            }
        }

        this.attachments = new ArrayList<byte[]>();

        this.attachments = this.getAttachmentsOutOfMimeMesage(mimeMessage);


    }
    private List<byte[]> getAttachmentsOutOfMimeMesage(MimeMessage message) throws IOException, MessagingException {

        MimeMessage tempMessage = message;
        List<byte[]> tempListAttachments = new ArrayList<byte[]>();

        if (tempMessage.getContent() instanceof Multipart) {
            getPartsMultipart((Multipart) tempMessage.getContent(), tempListAttachments);
        }
        else if (tempMessage.getContent() instanceof String)  {
            this.body = ((String) tempMessage.getContent()).trim();
        }

        return tempListAttachments;
    }
    private void getPartsMultipart(Multipart multipart, List<byte[]> lstAttachments) throws MessagingException, IOException {
        for (int i = 0; i < multipart.getCount(); i++) {

            BodyPart bodyPart = multipart.getBodyPart(i);
            if (bodyPart.getContent() instanceof Multipart) {
                getPartsMultipart((Multipart) bodyPart.getContent(), lstAttachments);
            }
            if (bodyPart.getContent() instanceof String) {
                this.body = (String) bodyPart.getContent();

            }
            else if (bodyPart.getContent() instanceof InputStream) {
                lstAttachments.add(IOUtils.toByteArray((InputStream) bodyPart.getContent()));
            }


        }
    }
    /**
     * @see org.unitils.mail.SmtpMessage#getBody()
     */
    @Override
    public String getBody() {
        return this.body;

    }

    /**
     * @return the from
     */
    public String getFrom() {
        return from;
    }

    /**
     * @return the subject
     */
    public String getSubject() {
        return subject;
    }

    /**
     * @return the to
     */
    public List<String> getTo() {
        return Collections.unmodifiableList(to);
    }

    /**
     * @return {@link List}
     */
    public List<byte[]> getAttachments() {
        return Collections.unmodifiableList(attachments);
    }

    /**
     * @return {@link List}
     */
    @Override
    public List<String> getBcc() {
        return Collections.unmodifiableList(bcc);
    }

    /**
     * @return {@link String}
     */
    public List<String> getCc() {
        return Collections.unmodifiableList(cc);
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("SmtpMessageImpl [from=" + from + ", subject=" + subject + ", body=" + body + ", ");
        if (to.size() > 0) {
            buffer.append("to=[");
            for (int i = 0; i < to.size() - 1; i++) {
                buffer.append(" " + to.get(i)  + ", ");
            }
            buffer.append(to.get(to.size() - 1));

            buffer.append("],");
        }


        if (cc.size() > 0) {
            buffer.append("cc=[");
            for (int i = 0; i < cc.size() - 1; i++) {
                buffer.append(" " + cc.get(i)  + ", ");
            }
            buffer.append(cc.get(cc.size() - 1));

            buffer.append("], "); 
        }

        if (bcc.size() > 0) {
            buffer.append("bcc=[");
            for (int i = 0; i < bcc.size() - 1; i++) {
                buffer.append(" " + bcc.get(i)  + ", ");
            }
            buffer.append(bcc.get(bcc.size() - 1));

            buffer.append("],"); 
        }


        buffer.append(" attachement size : " + attachments.size() + " ]");
        return buffer.toString() ;
    }


}