package org.unitils.objectvalidation;

import static org.unitils.objectvalidation.utils.Utils.checkNotNull;

import java.util.List;


/**
 * Object constructed by {@link ValidationModule} with the rules in the unitils properties file
 * or the rules defined in the {@link ObjectValidationRules} annotation.
 * 
 * @author Matthieu Mestrez
 * @since Oct 10, 2013
 */
public class ObjectValidator {
    
    private List<Rule> rulesToValidate;
    private List<EqualsHashCodeValidator> equalsHashCodeValidator;
    
    public ObjectValidator(List<Rule> rulesToValidate, List<EqualsHashCodeValidator> equalsHashCodeValidator) {
        checkNotNull(rulesToValidate, "The list of rules cannot be null");
        checkNotNull(equalsHashCodeValidator, "The equals and hashCode validator cannot be null");
        this.rulesToValidate = rulesToValidate;
        this.equalsHashCodeValidator = equalsHashCodeValidator;
    }

    public void validate(Class<?> classToValidate) {
        for (Rule rule : rulesToValidate) {
            rule.validate(classToValidate);
        }
    }

    public void validate(List<Class<?>> classesToValidate) {
        for (Class<?> classToValidate : classesToValidate) {
            validate(classToValidate);
        }

    }
    
    public void validate(Class<?>... classesToValidate) {
        for (Class<?> classToValidate : classesToValidate) {
            validate(classToValidate);
        }
    }

    public void validateEqualsAndHashCodeIgnoring(Class<?> classToValidate, String... fieldsToIgnore) {
        
        for (EqualsHashCodeValidator validator : equalsHashCodeValidator) {
            validator.validateIgnoring(classToValidate, fieldsToIgnore);
        }
        
    }

    public void validateEqualsAndHashCodeIgnoring(Class<?> classToValidate, List<String> fieldsToIgnore) {
        
        for (EqualsHashCodeValidator validator : equalsHashCodeValidator) {
            validator.validateIgnoring(classToValidate, fieldsToIgnore);
        }
        
    }

    public void validateEqualsAndHashCodeForFields(Class<?> classToValidate, List<String> onlyFields) {
        
        for (EqualsHashCodeValidator validator : equalsHashCodeValidator) {
            validator.validateOnly(classToValidate, onlyFields);
        }
        
    }

    public void validateEqualsAndHashCodeForFields(Class<?> classToValidate, String... onlyFields) {
        
        for (EqualsHashCodeValidator validator : equalsHashCodeValidator) {
            validator.validateOnly(classToValidate, onlyFields);
        }
        
    }

}
