package org.unitils.objectvalidation.rules;

import static org.junit.Assert.assertEquals;
import static org.unitils.objectvalidation.utils.Utils.checkNotNull;

import org.unitils.objectvalidation.ObjectCloner;
import org.unitils.objectvalidation.ObjectCreator;
import org.unitils.objectvalidation.Rule;


/**
 * Assert the compliancy of an object hashCode() method. Rules are the following : - Same objects must produce the same hashCode number. -
 * Equals object have the same number - Unequal numbers produce different hashcode number
 * 
 * @author Matthieu Mestrez
 * @since Oct 10, 2013
 */
public class HashCodeComplientRule implements Rule {

    private ObjectCreator randomFactory;
    private ObjectCloner objectCloner;

    public HashCodeComplientRule(ObjectCreator randomFactory, ObjectCloner objectCloner) {
        this.randomFactory = randomFactory;
        this.objectCloner = objectCloner;
    }

    @Override
    public void validate(Class<?> bean) {
        Object firstObject = randomFactory.createRandomObject(bean);
        Object equalsObject = objectCloner.deepClone(firstObject);
        Object differentObject = randomFactory.createRandomObject(bean);

        checkRules(firstObject, equalsObject, differentObject);
    }

    /**
     * @param firstInstance
     * @param secondInstance
     * @param thirdInstance
     */
    private void checkRules(Object firstObject, Object equalsObject, Object differentObject) {
        checkNotNull(firstObject, "firstInstance is not supposed to be null.");
        checkNotNull(equalsObject, "secondInstance is not supposed to be null.");
        checkNotNull(differentObject, "thirdInstance is not supposed to be null.");
        
        checkConsistency(firstObject);
        checkTwoEqualsObjectsProduceSameNumber(firstObject, equalsObject);
    }
    
    /** Checks that the hashCode is consistent with the same object. */
    public void checkConsistency(Object object) {
        int initialHashCode = object.hashCode();

        assertEquals("Consistent hashcode test fails", initialHashCode, object.hashCode());
        assertEquals("Consistent hashcode test fails", initialHashCode, object.hashCode());
    }

    /** Objects that are equal using the equals method should return the same integer. */
    public void checkTwoEqualsObjectsProduceSameNumber(Object firstObject, Object equalsObject) {
        int objectOneHashCode = firstObject.hashCode();
        int objectThreeHashCode = equalsObject.hashCode();

        assertEquals("Equal object, return equal hashcode test fails", objectOneHashCode, objectThreeHashCode);
    }

}
