package org.unitils.objectvalidation.utils;

import static org.unitils.objectvalidation.utils.Utils.checkNotNull;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.unitils.core.UnitilsException;


/**
 * Utility to visit a class hierarchy and return its tree node of classes.
 * 
 * @author Jeroen Horemans
 * @author Matthieu Mestrez
 * @since Oct 18, 2013
 */
public final class TreeNodeCreator {

    private TreeNodeCreator() { }
    
    public static final TreeNode createTreeNodeFor(Class<?> type) {
        TreeNode parentNode = new TreeNode(type);
        
        return fillInChildren(parentNode);
    }
    
    private static final TreeNode fillInChildren(TreeNode node) {
        
        checkNotNull(node, "The node cannot be null");
        
        Class<?> value = node.getValue();
        
        if (valueConsideredAsPrimitiveOrWithoutChildNodes(value)) {
            return node;
        }
        
        Constructor<?> constructor = figureOutConstructor(value, node);
        constructor.setAccessible(true);
        
        for (Class<?> parameterType : constructor.getParameterTypes()) {
            TreeNode child = new TreeNode(parameterType);
            node.addChild(child);
            fillInChildren(child);
        }
        
        return node;
    }

    private static boolean valueConsideredAsPrimitiveOrWithoutChildNodes(Class<?> value) {
        return  value.isEnum()
             || value.isInterface()
             || value.isArray()
             || value.isPrimitive()
             || Number.class.isAssignableFrom(value)
             || String.class.isAssignableFrom(value)
             || Collection.class.isAssignableFrom(value)
             || Boolean.class.isAssignableFrom(value)
             || Exception.class.isAssignableFrom(value);
    }

    private static Constructor<?> figureOutConstructor(Class<?> value, TreeNode parentNode) {
        
        List<Constructor<?>> constructors = Arrays.asList(value.getDeclaredConstructors());
        Collections.sort(constructors, new ConstructorSizeComparator());
        
        for (Constructor<?> constructor : constructors) {
            if (isCyclycDependencyOk(constructor, parentNode) && isNotGeneratedConstructor(constructor)) {
                return constructor;
            }
        }
        
        throw new UnitilsException("Could not create object, no constructor found for class " + value.getName() + " with creation tree " + parentNode);
    }

    private static boolean isNotGeneratedConstructor(Constructor<?> constructor) {
        if (constructor.getParameterTypes() != null && constructor.getParameterTypes().length >= 1) {
            return !constructor.getParameterTypes()[constructor.getParameterTypes().length - 1].getName().endsWith(constructor.getName() + "$1");
        }
        return true;
    }

    private static boolean isCyclycDependencyOk(Constructor<?> constructor, TreeNode parentNode) {
        
        for (Class<?> type : constructor.getParameterTypes()) {
            if (isClassAlreadyInTheTree(type, parentNode)) {
                return false;
            }
        }
        
        return true;
    }

    private static boolean isClassAlreadyInTheTree(Class<?> type, TreeNode node) {
        
        if (node == null) {
            return false;
        } else if (type.equals(node.getValue())) {
            return true;
        }
        
        return isClassAlreadyInTheTree(type, node.getParent());
    }
    
    /**
     * Sorts the constructor by their parameter size from the smallest to the biggest.
     */
    private static class ConstructorSizeComparator implements Comparator<Constructor<?>>, Serializable {

        private static final long serialVersionUID = -7354619453997861875L;

        @Override
        public int compare(Constructor<?> o1, Constructor<?> o2) {
            return o2.getParameterTypes().length - o1.getParameterTypes().length;
        }
    }

}
