package org.unitils.objectvalidation.objectcreator.generator;

import static junit.framework.Assert.fail;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Random;

/**
 * 
 * The primitive generator is capable of generating java primitive types and there counterparts. {@link Boolean}, {@link Integer},
 * {@link BigInteger}, {@link Float}, {@link Double}, {@link BigDecimal}, {@link Long}, {@link Short}, {@link Character}, {@link String},
 * {@link Date}, {@link java.sql.Date}, {@link Calendar}, {@link Timestamp}, {@link Byte}
 * 
 * @author Jeroen Horemans
 * @since Feb 20, 2012
 */
public class PrimitiveGenerator implements Generator {

    private Random random = new Random(new Date().getTime());

    private static final char[] CHARACTERS = new char[]{
        'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'
    };

    @Override
    public Object generateObject(Class<?> clazz, List<Object> input, List<Class<?>> inputClasses) throws Exception {
        return generateJavaDefaultClass(clazz);
    }

    protected Object generateJavaDefaultClass(final Class<?> type) {
        Object temp = testIfClassIsNumber(type);
        if (temp != null) {
            return temp;
        }
        
        temp = testIfClassContainsText(type);
        if (temp != null) {
            return temp;
        }
        temp = testIfClassIsDate(type);
        if (temp != null) {
            return temp;
        }
        temp = testIfClassIsException(type);
        if (temp != null) {
            return temp;
        }
 
        return null;
    }

    private Object testIfClassIsNumber(Class<?> type) {
        if (type == boolean.class || type == Boolean.class) {
            return random.nextBoolean();
        }

        if (type == int.class || type == Integer.class) {
            return random.nextInt();
        }

        if (type == BigInteger.class) {
            return BigInteger.valueOf(random.nextLong());
        }

        if (type == float.class || type == Float.class) {
            return random.nextFloat();
        }

        if (type == double.class || type == Double.class) {
            return random.nextDouble();
        }

        if (type == BigDecimal.class) {
            return BigDecimal.valueOf(random.nextDouble());
        }

        if (type == long.class || type == Long.class) {
            return random.nextLong();
        }

        if (type == short.class || type == Short.class) {
            return (short) (random.nextInt(Short.MAX_VALUE + 1) * (random.nextBoolean() ? 1 : -1));
        }

        if (type == byte.class || type == Byte.class) {
            byte[] randombyte = new byte[1];
            random.nextBytes(randombyte);
            return randombyte[0];
        }
        return null;
    }

    private Object testIfClassContainsText(Class<?> type) {
        if (type == char.class || type == Character.class) {
            return CHARACTERS[random.nextInt(CHARACTERS.length)];
        }

        if (type == String.class) {
            StringBuffer randomString = new StringBuffer(5);

            /* prevent zero length string lengths */
            for (int count = 0; count < random.nextInt(6 + 1) + 1; count++) {
                randomString.append(generateJavaDefaultClass(Character.class));
            }
            return randomString.toString();
        }
        return null;
    }
   
    private Object testIfClassIsDate(Class<?> type) {
        
        if (type == Date.class) {
            return new Date((Long) generateJavaDefaultClass(Long.class));
        }
        if (type == java.sql.Date.class) {
            return new java.sql.Date((Long) generateJavaDefaultClass(Long.class));
        }

        if (type == Timestamp.class) {
            return new Timestamp((Long) generateJavaDefaultClass(Long.class));
        }
        if (type == Calendar.class) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTimeInMillis((Long) generateJavaDefaultClass(Long.class));
            return calendar;
        }
        
        return null;
    }
    
    private Object testIfClassIsException(Class<?> type) {
        if (Exception.class.isAssignableFrom(type)) {
            try {
                return type.newInstance();
            } catch (InstantiationException e) {
                fail("Could not instantiate " + type.getName() + "\n" + e.getMessage());
            } catch (IllegalAccessException e) {
                fail("Could not instantiate " + type.getName() + "\n" + e.getMessage());
            }
        }
        return null;
    }

}
