package org.unitils.objectvalidation;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.fail;
import static org.unitils.util.ReflectionUtils.getFieldValue;
import static org.unitils.util.ReflectionUtils.setFieldValue;

import java.lang.reflect.Field;
import java.util.List;



/**
 * Validate the equals and hashCode method.
 * 
 * @author Matthieu Mestrez
 * @since Oct 31, 2013
 */
public class EqualsHashCodeValidator {

    private ObjectCreator objectCreator;
    private ObjectCloner objectCloner;

    public EqualsHashCodeValidator(ObjectCreator objectCreator, ObjectCloner objectCloner) {
        this.objectCreator = objectCreator;
        this.objectCloner = objectCloner;
    }

    void validate(Class<?> classToValidate, List<Field> fields) {
        Object randomObject = objectCreator.createRandomObject(classToValidate);
        Object copy = objectCloner.deepClone(randomObject);
        
        assertEquals("The object and its clone are expected to be equals", randomObject, copy);
        assertEquals("The object and its clone hashCodes are expected to be equals", randomObject.hashCode(), copy.hashCode());
        
        for (Field field : fields) {
            
                if (fieldCanBeCompared(field)) {
            
                Object randomField = objectCreator.createRandomObject(field.getType());
                
                int maxSearchForRandom = 1000;
                
                Object previousValue = getFieldValue(randomObject, field);
                while (previousValue.equals(randomField) && randomField != null) {
                    randomField = objectCreator.createRandomObject(field.getType());
                    maxSearchForRandom--;
                    if (maxSearchForRandom == 0) {
                        fail("The validator was unable to create a different value for the field " + field.getName() + " of the class " + classToValidate.getName() + " than " + randomField.toString());
                    }
                }

                setFieldValue(copy, field, randomField);
           
                assertFalse("Changing the field " + field.getName() + " of the class " + classToValidate.getName() + " from '" + previousValue +
                            "' to '" + randomField.toString() + "' still makes it an equals object. It is expected to be false.", randomObject.equals(copy));
                
                if (!field.getType().isPrimitive()) {
                    setFieldValue(copy, field, null);
                    
                    assertFalse("Changing the field " + field.getName() + " of the class " + classToValidate.getName() + " from '" + previousValue +
                        "' to 'null' still makes it an equals object. It is expected to be false.", randomObject.equals(copy));
                }
                
            }
        }
    }

    private boolean fieldCanBeCompared(Field field) {
        return field.getName() != "serialVersionUID"
            && field.getName() != "$jacocoData";
    }


}
