package org.unitils.objectvalidation;

import static java.lang.reflect.Modifier.isStatic;
import static junit.framework.Assert.assertFalse;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import static org.unitils.util.ReflectionUtils.getFieldValue;
import static org.unitils.util.ReflectionUtils.setFieldValue;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;



/**
 * Validate the equals and hashCode method.
 * 
 * This validator is created through the process of validation of an object : {@link ObjectValidator}.validate()
 * It has to receive a specific class and the list of fields that are in the equals and hashCode method.
 * 
 * It will generate a different object per field and compare them all to check if they are not equals.
 * 
 * @author Matthieu Mestrez
 * @since Oct 31, 2013
 */
public class EqualsHashCodeValidator {

    private ObjectCreator objectCreator;
    private ObjectCloner objectCloner;

    public EqualsHashCodeValidator(ObjectCreator objectCreator, ObjectCloner objectCloner) {
        this.objectCreator = objectCreator;
        this.objectCloner = objectCloner;
    }

    void validate(Class<?> classToValidate, List<Field> fields) {
        Object randomObject = objectCreator.createRandomObject(classToValidate);
        Object copy = objectCloner.deepClone(randomObject);
        
        List<Object> differentObjects = new ArrayList<Object>();
        
        
        assertEquals("The object and its clone are expected to be equals", randomObject, copy);
        assertEquals("The object and its clone hashCodes are expected to be equals", randomObject.hashCode(), copy.hashCode());
        
        for (Field field : fields) {
            
                if (!isStatic(field.getModifiers())) {
            
                    Object differentObject = createDifferentObjectThan(field, randomObject);
                    
                    differentObjects.add(differentObject);
                    
                    if (!field.getType().isPrimitive()) {
                        differentObjects.add(createObjectWithFieldNull(field, randomObject));
                    }
           
                
            }
        }
        
        for (Object object : differentObjects) {
            
            assertFalse("Objects were supposed to be different : \nObject 1 : " + randomObject + " \nObject 2 : " + object, randomObject.equals(object));
        }
    }
    
    private Object createDifferentObjectThan(Field field, Object randomObject) {
        Object randomField = objectCreator.createRandomObject(field.getGenericType());
        
        int maxSearchForRandom = 1000;
        Object differentObject = objectCloner.deepClone(randomObject);
        
        Object previousValue = getFieldValue(randomObject, field);
        while (sameValues(previousValue, randomField)) {
            randomField = objectCreator.createRandomObject(field.getGenericType());
            maxSearchForRandom--;
            if (maxSearchForRandom == 0) {
                fail("The validator was unable to create a different value for the field " + field.getName() + " of the class " + field.getDeclaringClass().getName() + " than " + randomField.toString());
            }
        }

        setFieldValue(differentObject, field, randomField);
        
        return differentObject;
    }
    
    private Object createObjectWithFieldNull(Field field, Object randomObject) {
        Object differentObject = objectCloner.deepClone(randomObject);
        
        setFieldValue(differentObject, field, null);
        
        return differentObject;
    }

    private boolean sameValues(Object previousValue, Object randomField) {
        return (primitiveEquals(previousValue, randomField)
            ||  arrayEquals(previousValue, randomField)
            || (previousValue.equals(randomField))) && randomField != null;
    }

    private boolean primitiveEquals(Object previousValue, Object randomField) {
        return previousValue.getClass().isPrimitive() && previousValue == randomField;
    }
    
    private boolean arrayEquals(Object previousValue, Object randomField) {
        if (previousValue.getClass().isArray()) {
            return Arrays.asList(previousValue).equals(Arrays.asList(randomField));
        }
        return false;
    }

}
