package org.unitils.objectvalidation.objectcreator;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.easymock.classextension.internal.objenesis.ObjenesisStd;
import org.unitils.core.UnitilsException;
import org.unitils.mock.core.MockObject;
import org.unitils.objectvalidation.ObjectCreator;
import org.unitils.objectvalidation.objectcreator.generator.Generator;
import org.unitils.objectvalidation.objectcreator.generator.GenericsGenerator;
import org.unitils.objectvalidation.objectcreator.generator.PrimitiveGenerator;
import org.unitils.objectvalidation.utils.ObjectCreatorTypeWrapper;
import org.unitils.util.ReflectionUtils;
/**
 * @author Matthieu Mestrez
 * @since 07/04/14.
 */
public class ObjectCreatorMockedFieldsImpl implements ObjectCreator {
    private static final Log LOGGER = LogFactory.getLog(ObjectCreatorMockedFieldsImpl.class);
    private GenericsGenerator genericsGenerator;
    private PrimitiveGenerator primitiveGenerator;

    private static Map<Class<?>, MockObject<?>> mocks = new HashMap<Class<?>, MockObject<?>>();;

    public ObjectCreatorMockedFieldsImpl() {
        primitiveGenerator = new PrimitiveGenerator();
        genericsGenerator = new GenericsGenerator();
    }




    public ObjectCreatorMockedFieldsImpl(GenericsGenerator genericsGenerator, PrimitiveGenerator primitiveGenerator) {
        super();
        this.genericsGenerator = genericsGenerator;
        this.primitiveGenerator = primitiveGenerator;
    }




    @Override
    public Object createRandomObject(Type type) {

        Object randomObject = null;

        ObjectCreatorTypeWrapper typeWrapper = new ObjectCreatorTypeWrapper(type);
        Object obj = createRealObject(typeWrapper);
        if (obj != null) {
            return obj;
        }
        if (typeWrapper.isClassType() || typeWrapper.isInterface()) {

            Class<?> clzz = typeWrapper.getClassOfType();
            randomObject = createInstance(typeWrapper);
            for (Field field : ReflectionUtils.getAllFields(clzz)) {
                if (!isConstant(field)) {
                    field.setAccessible(true);
                    Class<?> clazzField = null;
                    try {
                        clazzField = Class.forName(((Class<?>) field.getGenericType()).getName());
                    } catch (Exception e) {
                        clazzField = field.getType();
                    }

                    ReflectionUtils.setFieldValue(randomObject, field, createRealOrMockObject(new ObjectCreatorTypeWrapper(clazzField)));
                }
            }
        } else {
            throw new UnitilsException("Can only manage classes and interfaces. Abstract classes, generics, wildcard types are not handled by the mocked creator.");
        }

        return randomObject;
    }

    protected Map<Object, Object> createMap(ParameterizedType type) {
        Map<Object, Object> map = new HashMap<Object, Object>();
        Type[] typeArguments = type.getActualTypeArguments();
        Set<Class<?>> clzzesKey = genericsGenerator.getClassesFromTypeArguments(typeArguments[0]);
        Set<Class<?>> clzzesValue = genericsGenerator.getClassesFromTypeArguments(typeArguments[1]);

        for (int i = 0; i < new Random().nextInt(); i++) {
            Object objKey = pickClassFromSetAndGenerateAnObject(clzzesKey);
            Object objValue = pickClassFromSetAndGenerateAnObject(clzzesValue);

            map.put(objKey, objValue);
        }

        return map;
    }

    protected Set<Object> createSet(ParameterizedType type) {
        Set<Class<?>> clzzes = genericsGenerator.getClassesFromAParameterizedType(type);

        Set<Object> set = new HashSet<Object>();
        for (int i = 0; i < new Random().nextInt(); i++) {
            set.add(pickClassFromSetAndGenerateAnObject(clzzes));
        }

        return set;
    }

    protected Object pickClassFromSetAndGenerateAnObject(Set<Class<?>> clzzes) {
        if (clzzes.isEmpty()) {
            clzzes = genericsGenerator.getClassesFromTypeArguments(Object.class);
        }
        Class<?> clzz = new ArrayList<Class<?>>(clzzes).get(new Random().nextInt(clzzes.size()));
        return createRealOrMockObject(new ObjectCreatorTypeWrapper(clzz));
    }

    protected List<Object> createList(ParameterizedType type) {
        return new ArrayList<Object>(createSet(type));
    }

    protected <T> T[] createStuffedArray(Class<T> clzz) {
        int randomNumber = new Random().nextInt(10);
        Set<Class<?>> clzzes = genericsGenerator.getClassesFromTypeArguments(clzz);

        T[] arr = (T[]) Array.newInstance(clzz, randomNumber);
        for (int i = 0; i < randomNumber; i++) {
            arr[i] = (T) pickClassFromSetAndGenerateAnObject(clzzes);
        }

        return arr;
    }

    /**
     * Creates an object/mock of a specific class
     * @param typeWrapper
     * @return {@link Object}
     */
    Object createInstance(ObjectCreatorTypeWrapper typeWrapper) {
        Object temp = createRealObject(typeWrapper); 
        if (temp != null) {
            return temp;
        }
        Constructor<?> constructor = getConstructor(typeWrapper.getClassOfType());

        if (constructorNotEmpty(constructor)) {
            constructor.setAccessible(true);
            int i = 0;
            Class<?>[] classes = (Class[]) Array.newInstance(Class.class, constructor.getGenericParameterTypes().length);
            Object[] values = (Object[]) Array.newInstance(Object.class, constructor.getGenericParameterTypes().length);
            for (Type parameterType : constructor.getGenericParameterTypes()) {
                ObjectCreatorTypeWrapper type = new ObjectCreatorTypeWrapper(parameterType);
                classes[i] = type.getClassOfType();
                values[i] = createRealOrMockObject(type);
                i++;
            }
            return ReflectionUtils.createInstanceOfType(typeWrapper.getClassOfType(), true, classes, values);
        } else if (isDefaultConstructor(constructor)) {
            return ReflectionUtils.createInstanceOfType(typeWrapper.getClassOfType(), true);
        }

        return null;
    }

    /**
     * Some classes cannot be mocked. f.e. primitives and final objects.
     * @param typeWrapper 
     * @param clzz
     * @return
     */
    protected Object createRealObject(ObjectCreatorTypeWrapper typeWrapper ) {
        Class<?> clzz = typeWrapper.getClassOfType();
        if (isPrimitive(typeWrapper)) {
            try {
                return primitiveGenerator.generateObject(clzz, null, null, null);
            } catch (Exception e) {
                LOGGER.error(e.getMessage(), e);
                return null;
            }
        }

        if (Modifier.isFinal(clzz.getModifiers())) {
            return new ObjenesisStd().getInstantiatorOf(clzz).newInstance();
        }
        
        if (typeWrapper.isParameterizedType()) {
            ParameterizedType paramType =  (ParameterizedType) typeWrapper.getWrappedType();
            ObjectCreatorTypeWrapper rawTypeWrapper =  new ObjectCreatorTypeWrapper(paramType.getRawType());
            
            if (rawTypeWrapper.isMap()) {
                return createMap(paramType);
            } else if (rawTypeWrapper.isSet()) {
                return createSet(paramType);
            } else if (rawTypeWrapper.isList()) {
                return createList(paramType);
            }


        } else if (typeWrapper.isArray()) {
            return createStuffedArray(typeWrapper.getClassOfType());
        } 
        
        return null;
    }
   


    private MockObject<Object> mockObj;

    protected Object createRealOrMockObject(ObjectCreatorTypeWrapper typeWrapper) {
        Object realObject = createRealObject(typeWrapper);
        if (realObject != null) {
            return realObject;
        } 
        Class<?> clzz = typeWrapper.getClassOfType();

        if (mocks.containsKey(clzz)) {
            MockObject<?> mock = mocks.get(clzz);
            mock.resetBehavior();
            return mock.getObjectToInject();
        }
        mockObj = new MockObject(clzz, this);
        mockObj.resetBehavior();
        Object obj = null;
        mockObj.returns(true).equals(obj);
        //mockObj.resetBehavior();
        //mockObj.returns(1).hashCode();
        Object mock = mockObj.getObjectToInject();
        mocks.put(clzz, mockObj);
        return mock;
    }

    /**
     * This method checks if this constructor is the default constructor.
     * @param constructor
     * @return boolean
     */
    protected boolean constructorNotEmpty(Constructor<?> constructor) {
        return constructor != null && constructor.getParameterTypes().length > 0;
    }
    /**
     * Returns true if this is the default constructor.
     * @param constructor
     * @return {@link Boolean}
     */
    protected boolean isDefaultConstructor(Constructor<?> constructor) {
        return constructor != null && constructor.getParameterTypes().length == 0;
    }

    /**
     * This method gets the first constructor that isn't generated by the compiler.
     * @param clzz
     * @return {@link Constructor}
     */
    protected Constructor<?> getConstructor(Class<?> clzz) {
        for (Constructor<?> constructor : clzz.getConstructors()) {
            if (!constructor.isSynthetic()) {
                return constructor;
            }
        }

        return null;
    }

    /**
     * @see org.unitils.objectvalidation.ObjectCreator#addGenerators(org.unitils.objectvalidation.objectcreator.generator.Generator[])
     */
    @Override
    public void addGenerators(Generator... generator) {
        //there are no generators needed.
        //This objectcreator creates mocks, no real objects.

    }

    /**
     * This method checks if the wrapper is a primitive.
     * @param wrapper
     * @return boolean
     */
    protected boolean isPrimitive(ObjectCreatorTypeWrapper wrapper) {
        return wrapper.isPrimitive() || wrapper.isString();
    }

    protected boolean isConstant(Field field) {
        return Modifier.isFinal(field.getModifiers()) && Modifier.isStatic(field.getModifiers());
    }


}
