package org.unitils.objectvalidation.objectcreator.generator;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.easymock.classextension.internal.objenesis.ObjenesisStd;
import org.reflections.ReflectionUtils;
import org.reflections.Reflections;
import org.reflections.ReflectionsException;
import org.reflections.scanners.SubTypesScanner;
import org.unitils.objectvalidation.objectcreator.generator.helper.ClassPathHelperExtended;
import org.unitils.objectvalidation.utils.TreeNode;

import sun.reflect.generics.reflectiveObjects.WildcardTypeImpl;

import com.google.common.base.Predicate;

/**
 * Creates objects for all kind of objects that use generics.
 *
 * @author Willemijn Wouters
 *
 * @since 1.1.6
 *
 */
public class GenericsGenerator implements Generator {

    private static final Log LOGGER = LogFactory.getLog(GenericsGenerator.class);

    private final Reflections reflections;

    private ObjenesisStd objenesisStd;

    public GenericsGenerator() {
        reflections = new Reflections(new SubTypesScanner(false), ClassPathHelperExtended.forProject(this.getClass().getClassLoader()));
        objenesisStd = new ObjenesisStd();
    }

    /**
     * @see
     * org.unitils.objectvalidation.objectcreator.generator.Generator#generateObject(java.lang.Class, java.util.List, java.util.List, java.util.List) 
     */
    @Override
    public Object generateObject(Class<?> clazz, List<Object> input, List<Class<?>> inputClasses, List<TreeNode> genericSubTypes)  {
        if (clazz.isAssignableFrom(ParameterizedType.class) || clazz.isAssignableFrom(WildcardTypeImpl.class)) {
            Set<Class<?>> tempClasses = new HashSet<Class<?>>();
            for (TreeNode treeNode : genericSubTypes) {
                tempClasses.addAll(getClassesFromAParameterizedType(treeNode.getParameterizedType()));
            }

            return pickAndCreateAnObjectFromASet(tempClasses);
        }
        return null;

    }

    /**
     * Get all the sub/super classes from a the {@link ParameterizedType}.
     * @param type: This is the {@link ParameterizedType} from a generic.
     * @return {@link Set}
     */
    public Set<Class<?>> getClassesFromAParameterizedType(ParameterizedType type) {
        Set<Class<?>> classesParamType = new HashSet<Class<?>>();
        if (type != null) {

            Type[] typeArg = type.getActualTypeArguments();
            for (Type type2 : typeArg) {
                classesParamType.addAll(getClassesFromTypeArguments(type2));
            }
        }
        return classesParamType;
    }

    /**
     * Get the classes from a specific type.
     * The type can be just a class or can be {@link WildcardTypeImpl}.
     * @param type: Get all the super/subtypes from a specific {@link Type}.
     * @return {@link Set}
     */
    public Set<Class<?>> getClassesFromTypeArguments(Type type) {
        Set<Class<?>> classesParamType = new HashSet<Class<?>>();
        //check if the generics has a wildcard (--> extends/super)
        if (type instanceof WildcardTypeImpl) {
            WildcardTypeImpl wildCard = (WildcardTypeImpl) type;
            classesParamType = (Set<Class<?>>) getClassesOfSubtypes(wildCard, classesParamType);
            classesParamType = getClassesOfSuperType(wildCard, classesParamType);
        } else {
            classesParamType.add((Class<?>) type);
        }

        return classesParamType;
    }

    /**
     * This method creates a random object from a {@link Set} of classes.
     *
     * @param collClasses: This is a collection of classes.
     * @return {@link Object}
     */
    protected Object pickAndCreateAnObjectFromASet(Set<Class<?>> collClasses) {
        try {
            if (!CollectionUtils.isEmpty(collClasses) && !(collClasses.size() == 1 && collClasses.contains(Object.class))) {
                return createRandomObjects(collClasses);
            }
        } catch (Exception e) {
            LOGGER.error("Something went wrong with creating one of the genrerics", e);
            //So let's try to create a plain object.
        }

        //Oops, something went wrong or there was just a wildcard.
        //The following lines gets all the loadable classes and picks one class.
        Set<Class<? extends Object>> asList = reflections.getSubTypesOf(Object.class);
        asList = getSetWithClassesThatOverridesASpecificMethod(asList, "equals", Object.class);
        asList = getSetWithClassesThatOverridesASpecificMethod(asList, "hashCode");
        return createRandomObjects(asList);
    }

    /**
     * Generate the exact object or a subtype from a specific class.
     *
     * @param classForType: the generator should look for a subtype of this class.
     * @return {@link Object}
     */
    public Object generateSubTypesObject(Class<?> classForType) {
        Set<Class<?>> allClasses = new HashSet<Class<?>>();
        allClasses = (Set<Class<?>>) getClassesOfSubtypes(classForType, allClasses);
        if (allClasses.isEmpty() && !classForType.isInterface() && !Modifier.isAbstract(classForType.getModifiers())) {
            //There are no subtypes, so let's add the parameter.
            allClasses.add(classForType);
        }
        return pickAndCreateAnObjectFromASet(allClasses);

    }

    /**
     * Goes through all the objects in a set and removes all the classes that
     * doesn't override a specific method.
     *
     * @param set: A set of classes.
     * @param methodName: the method that this method should find
     * @param parameterTypes: the parameters of the method that the classes must override
     * @return {@link Set}
     */
    protected Set<Class<? extends Object>> getSetWithClassesThatOverridesASpecificMethod(Set<Class<? extends Object>> set, String methodName, Class<?>... parameterTypes) {
        Set<Class<? extends Object>> temp = new HashSet<Class<? extends Object>>(set);

        for (Class<? extends Object> clzz : set) {
            try {
                Method method = clzz.getMethod(methodName, parameterTypes);

                if (!method.getDeclaringClass().equals(clzz)) {
                    temp.remove(clzz);
                }
            } catch (NoSuchMethodException e) {
                // do nothing
            } catch (SecurityException e) {
                // do nothing
            }
        }

        return temp;
    }

    /**
     * Get the subtypes of a class (based on a wildcard).
     *
     * @param wildcard: the wildcard of the generic
     * @param allClasses: A collection where the {@link Generator} should add the found classes.
     * @return {@link Set}
     */
    protected Set<?> getClassesOfSubtypes(WildcardTypeImpl wildcard, Set<?> allClasses) {

        for (Type type : wildcard.getUpperBounds()) {
            allClasses = getClassesOfSubtypes((Class<?>) type, allClasses);
        }

        return allClasses;
    }

    /**
     * Get the subtypes of a {@link Class}
     *
     * @param clzz: look for subtypes of this {@link Class}.
     * @param allClasses: A collection where the {@link Generator} should add the found classes.
     * @return {@link Set}
     */
    protected Set<?> getClassesOfSubtypes(Class<?> clzz, Set<?> allClasses) {
        try {

            Set<?> tempSubtypes = reflections.getSubTypesOf(clzz);
            if (CollectionUtils.isEmpty(allClasses)) {
                allClasses = tempSubtypes;
            } else {
                allClasses = compareSubtypesSets(allClasses, tempSubtypes);
            }
        } catch (ReflectionsException e) {
            //something went wrong with finding a subtype.
        }
        return allClasses;
    }

    /**
     * An {@link WildcardTypeImpl} can have multiple lowerbounds (super).
     *
     * @param wildcard: the wildcard of the generic
     * @param allClasses: A collection where the {@link Generator} should add the found classes.
     * @return {@link Set}
     */
    protected Set<Class<?>> getClassesOfSuperType(WildcardTypeImpl wildcard, Set<Class<?>> allClasses) {
        for (Type type : wildcard.getLowerBounds()) {
            allClasses = getClassesOfSuperType((Class<?>) type, allClasses);
        }

        return allClasses;
    }

    /**
     * Get all the supertypes from a {@link Class}
     *
     * @param clzz: look for supertypes of this {@link Class}.
     * @param allClasses: A collection where the {@link Generator} should add the found classes.
     * @return {@link Set}
     */
    protected Set<Class<?>> getClassesOfSuperType(Class<?> clzz, Set<Class<?>> allClasses) {
        Set<Class<?>> tempSuperTypes = ReflectionUtils.getAllSuperTypes(clzz, (Predicate<? super Class<?>>[]) null);
        if (CollectionUtils.isEmpty(allClasses)) {
            allClasses = tempSuperTypes;
        } else {
            allClasses = compareSubtypesSets(allClasses, tempSuperTypes);
        }
        return allClasses;
    }

    /**
     * This method creates a random object of a random class found with {@link GenericsGenerator#getClassesOfSubtypes(WildcardTypeImpl, Set)} or {@link GenericsGenerator#getClassesOfSuperType(WildcardTypeImpl, Set)}.
     *
     * @param classes: A set with a collection of classes.
     * @return {@link Object}
     */
    protected Object createRandomObjects(Set<Class<?>> classes) {
        List<Class<?>> tempClasses = new ArrayList<Class<?>>();
        //filter out all interfaces --> can't generate objects of interfaces unless there is a custom generator.
        //just to be sure try to remove all the interfaces.
        for (Class<?> clzz : classes) {
            if (!clzz.isInterface()) {
                tempClasses.add(clzz);
            }
        }

        if (tempClasses.isEmpty()) {
            //Apparantly there are only interfaces, just hope that the user has created a custom generator.
            tempClasses.addAll(classes);
        }

        Random random = new Random();

        Class<?> obj = tempClasses.get(random.nextInt(tempClasses.size()));

        return objenesisStd.getInstantiatorOf(obj).newInstance();
    }

    /**
     * Returns only the in common classes of set1 and set2
     *
     * @param set1: an object of type {@link Set}
     * @param set2: an object of type {@link Set}
     * @return {@link Set}
     */
    public Set<Class<?>> compareSubtypesSets(Set<?> set1, Set<?> set2) {
        return new HashSet<Class<?>>(CollectionUtils.intersection(set1, set2));
    }
}
