package org.unitils.objectvalidation;

import static java.util.Arrays.asList;
import static org.junit.Assert.assertNotNull;
import static org.unitils.objectvalidation.utils.Utils.checkNotNull;
import static org.unitils.util.ReflectionUtils.getAllFields;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.unitils.objectvalidation.ValidationResult.ValidationResultBuilder;

/**
 * Object constructed by {@link ValidationModule} with the rules in the unitils
 * properties file or the rules defined in the {@link ObjectValidationRules}
 * annotation.
 *
 * @author Matthieu Mestrez
 * @since Oct 10, 2013
 */
public class ObjectValidator {

    private List<Rule> rulesToValidate;
    private List<EqualsHashCodeValidator> equalsHashCodeValidators;

    public ObjectValidator(List<Rule> rulesToValidate, List<EqualsHashCodeValidator> equalsHashCodeValidators) {
        checkNotNull(rulesToValidate, "The list of rules cannot be null");
        checkNotNull(equalsHashCodeValidators, "The equals and hashCode validator cannot be null");
        this.rulesToValidate = rulesToValidate;
        this.equalsHashCodeValidators = equalsHashCodeValidators;
    }

    private void validate(ClassToValidate classToValidate) {
        ValidationResult validationResult = doValidate(classToValidate);

        if (!validationResult.isValid()) {
            throw new AssertionError(validationResult.toString());
        }
    }

    private ValidationResult doValidate(ClassToValidate classToValidate) {
        ValidationResultBuilder validationResultBuilder = ValidationResult.buildValidationResultForClass(classToValidate.getClassToValidate());

        for (Rule rule : rulesToValidate) {

            try {
                rule.validate(classToValidate.getClassToValidate());
            } catch (Throwable throwable) {
                throwable.printStackTrace();
                validationResultBuilder.andMessage(new ValidationMessage(rule.getClass(), ExceptionUtils.getFullStackTrace(throwable)));
            }
        }

        if (classToValidate.hasToValidateEqualsAndHashCode()) {
            for (EqualsHashCodeValidator equalsHashCodeValidator : equalsHashCodeValidators) {
                try {
                    equalsHashCodeValidator.validate(classToValidate.getClassToValidate(), classToValidate.getFieldsToValidate());
                } catch (Throwable throwable) {
                    validationResultBuilder.andMessage(new ValidationMessage(equalsHashCodeValidator.getClass(), ExceptionUtils.getFullStackTrace(throwable)));
                }
            }
        }

        return validationResultBuilder.build();
    }

    public ValidateEqualsAndHashCode classToValidate(Class<?> classToValidate) {
        return new BuildValidator(classToValidate, this);
    }

    public interface ValidateEqualsAndHashCode {

        ChooseFieldsInEquals checkingAllPossibilities();

        AddClassToValidate withoutCheckingAllPossibilities();
    }

    public interface ChooseFieldsInEquals {

        AddClassToValidate ignoringFields(String... fieldNames);

        AddClassToValidate withFieldNames(String... fieldNames);

        AddClassToValidate withAllFields();
    }

    public interface AddClassToValidate {

        ValidateEqualsAndHashCode classToValidate(Class<?> classToValidate);

        void validate();
    }

    public class BuildValidator implements ValidateEqualsAndHashCode, ChooseFieldsInEquals, AddClassToValidate {

        private List<ClassToValidate> classesToValidate;
        private ClassToValidate classToValidateToAdd;
        private ObjectValidator objectValidator;

        BuildValidator(Class<?> classToValidate, ObjectValidator objectValidator) {
            assertNotNull(classToValidate);
            assertNotNull("The objectValidator seems not initialized, @RunWith(UnitilsJUnit4TestClassRunner.class) and unitils.properties configuration has to be done", objectValidator);
            this.classesToValidate = new ArrayList<ObjectValidator.ClassToValidate>();
            this.classToValidateToAdd = new ClassToValidate(classToValidate);
            this.objectValidator = objectValidator;
        }

        @Override
        public AddClassToValidate withFieldNames(String... fieldNames) {
            List<Field> fields = new ArrayList<Field>();
            List<String> fieldsToValidate = asList(fieldNames);

            for (Field field : getAllFields(classToValidateToAdd.getClassToValidate())) {
                if (fieldsToValidate.contains(field.getName())) {
                    fields.add(field);
                }
            }

            classToValidateToAdd.setFieldsToValidate(fields);
            classesToValidate.add(classToValidateToAdd);
            return this;
        }

        @Override
        public AddClassToValidate ignoringFields(String... fieldNames) {
            List<Field> fields = new ArrayList<Field>();
            List<String> fieldsToValidate = asList(fieldNames);

            for (Field field : getAllFields(classToValidateToAdd.getClassToValidate())) {
                if (!fieldsToValidate.contains(field.getName())) {
                    fields.add(field);
                }
            }

            classToValidateToAdd.setFieldsToValidate(fields);
            classesToValidate.add(classToValidateToAdd);
            return this;
        }

        @Override
        public AddClassToValidate withAllFields() {
            List<Field> fields = new ArrayList<Field>();

            for (Field field : getAllFields(classToValidateToAdd.getClassToValidate())) {
                fields.add(field);
            }

            classToValidateToAdd.setFieldsToValidate(fields);
            classesToValidate.add(classToValidateToAdd);
            return this;
        }

        @Override
        public ValidateEqualsAndHashCode classToValidate(Class<?> classToValidate) {
            this.classToValidateToAdd = new ClassToValidate(classToValidate);
            return this;
        }

        @Override
        public void validate() {
            for (ClassToValidate classToValidate : classesToValidate) {
                objectValidator.validate(classToValidate);
            }
        }

        @Override
        public AddClassToValidate withoutCheckingAllPossibilities() {
            classToValidateToAdd.dontValidateEqualsAndHashCode();
            classesToValidate.add(classToValidateToAdd);
            return this;
        }

        @Override
        public ChooseFieldsInEquals checkingAllPossibilities() {
            classToValidateToAdd.validateEqualsAndHashCode();
            return this;
        }
    }

    private static class ClassToValidate {

        private Class<?> classToValidate;
        private List<Field> fieldsToValidate;
        private boolean validateEqualsAndHashCode;

        public ClassToValidate(Class<?> classToValidate) {
            this.classToValidate = classToValidate;
        }

        public void dontValidateEqualsAndHashCode() {
            this.validateEqualsAndHashCode = false;
        }

        public boolean hasToValidateEqualsAndHashCode() {
            return validateEqualsAndHashCode;
        }

        public void validateEqualsAndHashCode() {
            this.validateEqualsAndHashCode = true;
        }

        public void setFieldsToValidate(List<Field> fieldsToValidate) {
            this.fieldsToValidate = fieldsToValidate;
        }

        public Class<?> getClassToValidate() {
            return classToValidate;
        }

        public List<Field> getFieldsToValidate() {
            return fieldsToValidate;
        }

    }

}
