package org.unitils.objectvalidation.objectcreator.generator;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.unitils.objectvalidation.ObjectCreator;
import org.unitils.objectvalidation.objectcreator.generator.helper.GeneratorHelper;
import org.unitils.objectvalidation.utils.ObjectCreatorTypeWrapper;
import org.unitils.objectvalidation.utils.TreeNode;


/**
 * Create objects of type {@link ParameterizedType}.
 *
 * @author Willemijn Wouters
 *
 * @since 1.1.8
 *
 * @see <a href="http://docs.oracle.com/javase/7/docs/api/java/lang/reflect/ParameterizedType.html">ParameterizedType</a>
 *
 */
public class ParameterizedTypeGenerator implements Generator {

    private ObjectCreator objectCreator;

    public ParameterizedTypeGenerator(ObjectCreator objectCreator) {
        this.objectCreator = objectCreator;
    }

    /**
     * @see org.unitils.objectvalidation.objectcreator.generator.Generator#generateObject(java.lang.Class, java.util.List, java.util.List, java.util.List)
     */
    @Override
    public Object generateObject(Class<?> clazz, List<Object> input, List<Class<?>> inputClasses, List<TreeNode> genericSubTypes) throws Exception {
        if (ParameterizedType.class.isAssignableFrom(clazz)  && !CollectionUtils.isEmpty(genericSubTypes)) {
            return createParameterizedType((ParameterizedType) genericSubTypes.get(0).getType(), input, inputClasses);
        }
        return null;
    }
    /**
     * This method generates an object of type {@link ParameterizedType}.
     * @param type : the {@link ParameterizedType}
     * @param input : a list with input objects for the {@link CollectionGenerator}
     * @param inputClasses : a list of input classes for the {@link CollectionGenerator}.
     * @return {@link Object}
     * @throws Exception
     */
    public Object createParameterizedType(ParameterizedType type, List<Object> input, List<Class<?>> inputClasses) throws Exception {
        ObjectCreatorTypeWrapper typeWrapper = new ObjectCreatorTypeWrapper(type.getRawType());
        if (isCollection(typeWrapper)) {
            TreeNode node = GeneratorHelper.createTreenode(type.getRawType());
            List<TreeNode> nodes = new ArrayList<TreeNode>();
            for (Type subtype : type.getActualTypeArguments()) {
                nodes.add(GeneratorHelper.createTreenode(subtype));
            }
            node.setGenericSubtype(nodes);
            return objectCreator.getGeneratorOfType(CollectionGenerator.class).generateObject((Class<?>) type.getRawType(), input, inputClasses, nodes);

        }
        //TODO: write this piece + test what happens when their is f.e. another parameterized type inside.
        return objectCreator.createRandomObject(GeneratorHelper.createTreenode(type.getRawType()));
    }

    /**
     * This method checks if the type is a collection or not.
     * @param typeWrapper : an {@link ObjectCreatorTypeWrapper}.
     * @return boolean
     */
    protected boolean isCollection(ObjectCreatorTypeWrapper typeWrapper) {
        return typeWrapper.isArray() || typeWrapper.isList() || typeWrapper.isMap() || typeWrapper.isSet();
    }
}
