package org.unitils.objectvalidation;

import static org.unitils.objectvalidation.utils.Utils.checkNotNull;
import static org.unitils.objectvalidation.utils.Utils.findOneAndOnlyFieldForAnnotation;
import static org.unitils.objectvalidation.utils.Utils.toEqualsHashCodeValidator;
import static org.unitils.objectvalidation.utils.Utils.toRulesCollection;
import static org.unitils.util.ReflectionUtils.setFieldValue;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.unitils.core.TestListener;


/**
 * Contains the logic to load the validation logic from the test and run it.
 * 
 * @author Matthieu Mestrez
 * @since Oct 9, 2013
 */
public class ValidationModuleTestListener extends TestListener {

    private ObjectValidationRulesCollection validationRulesCollection;
    
    public ValidationModuleTestListener(ObjectValidationRulesCollection validationRulesCollection) {
        checkNotNull(this.getClass().getName() + " must have validationRules", validationRulesCollection);
        this.validationRulesCollection = validationRulesCollection;
    }
    
    /**
     * @see org.unitils.core.TestListener#beforeTestMethod(java.lang.Object, java.lang.reflect.Method)
     */
    @Override
    public void beforeTestMethod(Object testObject, Method testMethod) {
        prepareObjectValidatorFieldIn(testObject);
        super.afterCreateTestObject(testObject);
    }
   

    private void prepareObjectValidatorFieldIn(Object testClass) {
        List<Field> objectValidatorFields = findOneAndOnlyFieldForAnnotation(testClass.getClass(), ObjectValidationRules.class, ObjectValidator.class);
        
        for (Field objectValidatorField : objectValidatorFields) {
            
            ObjectValidator validator = createObjectValidator(objectValidatorField);
            
            setFieldValue(testClass, objectValidatorField, validator);
        }
        
    }
    
	private ObjectValidator createObjectValidator(Field objectValidatorField) {
        List<Rule> rulesToValidate = buildRulesForField(objectValidatorField);
        
        List<EqualsHashCodeValidator> equalsHashCodeValidator = buildEqualsHashCodeValidator(objectValidatorField);
        
        return new ObjectValidator(rulesToValidate, equalsHashCodeValidator);
    }
    
    private List<Rule> buildRulesForField(Field field) {
        ObjectValidationRules annotation = field.getAnnotation(ObjectValidationRules.class);
        
        List<Rule> rules = new ArrayList<Rule>();
        
        if (annotation.replacementRules() != null && annotation.replacementRules().length > 0) {
            rules.addAll(constructRules(annotation.replacementRules()));
        } else {
            rules.addAll(validationRulesCollection.getRules());
        }
        
        if (annotation.additionalRules() != null && annotation.additionalRules().length > 0) {
            rules.addAll(constructRules(annotation.additionalRules()));
        }
        
        return rules;
    }

    private List<EqualsHashCodeValidator> buildEqualsHashCodeValidator(Field field) {
        ObjectValidationRules annotation = field.getAnnotation(ObjectValidationRules.class);
        List<EqualsHashCodeValidator> validator = new ArrayList<EqualsHashCodeValidator>();
        if (annotation.replacementRules() != null && annotation.replacementRules().length > 0) {
            validator.addAll(constructEqualsHashCodeValidator(annotation.replacementRules()));
        } else {
            validator.add(validationRulesCollection.getEqualsHashCodeValidator());
        }
        if (annotation.additionalRules() != null && annotation.additionalRules().length > 0) {
            validator.addAll(constructEqualsHashCodeValidator(annotation.additionalRules()));
        }
        
        return validator;
    }

    private List<Rule> constructRules(Class<? extends ObjectValidationRulesCollection>[] rules) {
        List<Rule> objectValidationRules = new ArrayList<Rule>();
        for (Class<? extends ObjectValidationRulesCollection> ruleCollection : rules) {
            objectValidationRules.addAll(toRulesCollection(ruleCollection).getRules());
        }
        return objectValidationRules;
    }
    
    private List<EqualsHashCodeValidator> constructEqualsHashCodeValidator(Class<? extends ObjectValidationRulesCollection>[] rules) {
        List<EqualsHashCodeValidator> validators = new ArrayList<EqualsHashCodeValidator>();
        for (Class<? extends ObjectValidationRulesCollection> ruleCollection : rules) {
            validators.add(toEqualsHashCodeValidator(ruleCollection));
        }
        
        return validators;
    }

}
