package org.unitils.objectvalidation.objectcreator.generator.helper;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.easymock.classextension.internal.objenesis.ObjenesisStd;
import org.reflections.ReflectionUtils;
import org.reflections.Reflections;
import org.reflections.ReflectionsException;
import org.reflections.scanners.SubTypesScanner;
import org.unitils.objectvalidation.reflect.TypeWrapper;
import org.unitils.objectvalidation.utils.TreeNode;

import com.google.common.base.Predicate;


/**
 * This class generates supertypes or subtypes from a specific class.
 *
 * @author Willemijn Wouters
 *
 * @since 1.1.8
 *
 */
public class GeneratorHelper {
    private static final Log LOGGER = LogFactory.getLog(GeneratorHelper.class);

    /**
     * Get all the supertypes from a {@link Class}
     *
     * @param clzz: look for supertypes of this {@link Class}.
     * @param allClasses: A collection where the {@link org.unitils.objectvalidation.objectcreator.generator.Generator} should add the found classes.
     * @return {@link Set}
     */
    public static Set<Class<?>> getClassesOfSuperType(Class<?> clzz, Set<Class<?>> allClasses) {
        return getInstance().getClassesOfSuperTypeInstance(clzz, allClasses);
    }

    /**
     * Returns only the in common classes of set1 and set2
     *
     * @param set1: an object of type {@link Set}
     * @param set2: an object of type {@link Set}
     * @return {@link Set}
     */
    public static Set<Class<?>> compareSubtypesSets(Set<?> set1, Set<?> set2) {
        return new HashSet<Class<?>>(CollectionUtils.intersection(set1, set2));
    }

    /**
     * Generate the exact object or a subtype from a specific class.
     *
     * @param classForType: the generator should look for a subtype of this class.
     * @return {@link Object}
     */
    public static Object generateSubTypesObject(Class<?> classForType) {
        return getInstance().generateSubTypesObjectInstance(classForType);

    }



    /**
     * Get the subtypes of a {@link Class}
     *
     * @param clzz: look for subtypes of this {@link Class}.
     * @param allClasses: A collection where the {@link org.unitils.objectvalidation.objectcreator.generator.Generator} should add the found classes.
     * @return {@link Set}
     */
    public static Set<?> getClassesOfSubtypes(Class<?> clzz, Set<?> allClasses) {
        return getInstance().getClassesOfSubtypesInstance(clzz, allClasses);
    }

    /**
     * Goes through all the objects in a set and removes all the classes that
     * doesn't override a specific method.
     *
     * @param set: A set of classes.
     * @param methodName: the method that this method should find
     * @param parameterTypes: the parameters of the method that the classes must override
     * @return {@link Set}
     */
    public static Set<Class<? extends Object>> getSetWithClassesThatOverridesASpecificMethod(Set<Class<? extends Object>> set, String methodName, Class<?>... parameterTypes) {
        Set<Class<? extends Object>> temp = new HashSet<Class<? extends Object>>(set);

        for (Class<? extends Object> clzz : set) {
            try {
                Method method = clzz.getMethod(methodName, parameterTypes);

                if (!method.getDeclaringClass().equals(clzz)) {
                    temp.remove(clzz);
                }
            } catch (NoSuchMethodException e) {
                LOGGER.error(e.getMessage(), e);
            } catch (SecurityException e) {
                LOGGER.error(e.getMessage(), e);
            }
        }

        return temp;
    }


    /**
     * This method creates a random object from a {@link Set} of classes.
     *
     * @param collClasses: This is a collection of classes.
     * @return {@link Object}
     */
    public static Object pickAndCreateAnObjectFromASet(Set<Class<?>> collClasses) {
        return getInstance().pickAndCreateAnObjectFromASetInstance(collClasses);
    }


    private static GeneratorHelper subSuperTypes;
    private ObjenesisStd objenesisStd;
    private final Reflections reflections;


    public static GeneratorHelper getInstance() {
        if (subSuperTypes == null) {
            subSuperTypes = new GeneratorHelper();
        }
        return subSuperTypes;
    }

    private GeneratorHelper() {
        reflections = new Reflections(new SubTypesScanner(false), ClassPathHelperExtended.forProject(this.getClass().getClassLoader()));
        objenesisStd = new ObjenesisStd();
    }

    protected Set<Class<?>> getClassesOfSuperTypeInstance(Class<?> clzz, Set<Class<?>> allClasses) {
        Set<Class<?>> tempSuperTypes = ReflectionUtils.getAllSuperTypes(clzz, (Predicate<? super Class<?>>[]) null);
        if (CollectionUtils.isEmpty(allClasses)) {
            allClasses = tempSuperTypes;
        } else {
            allClasses = compareSubtypesSets(allClasses, tempSuperTypes);
        }
        return allClasses;
    }

    protected Object generateSubTypesObjectInstance(Class<?> classForType) {
        Set<Class<?>> allClasses = new HashSet<Class<?>>();
        allClasses = (Set<Class<?>>) getClassesOfSubtypes(classForType, allClasses);
        if (allClasses.isEmpty() && !classForType.isInterface() && !Modifier.isAbstract(classForType.getModifiers())) {
            //There are no subtypes, so let's add the parameter.
            allClasses.add(classForType);
        }
        return pickAndCreateAnObjectFromASet(allClasses);
    }

    protected Set<?> getClassesOfSubtypesInstance(Class<?> clzz, Set<?> allClasses) {
        try {

            Set<?> tempSubtypes = reflections.getSubTypesOf(clzz);
            if (CollectionUtils.isEmpty(allClasses)) {
                allClasses = tempSubtypes;
            } else {
                allClasses = compareSubtypesSets(allClasses, tempSubtypes);
            }
        } catch (ReflectionsException e) {
            LOGGER.error("Something went wrong with finding a subtype", e);
        }
        return allClasses;
    }

    protected Object pickAndCreateAnObjectFromASetInstance(Set<Class<?>> collClasses) {
        try {
            if (!CollectionUtils.isEmpty(collClasses) && !(collClasses.size() == 1 && collClasses.contains(Object.class))) {
                return createRandomObjects(collClasses);
            }
        } catch (Exception e) {
            LOGGER.error("Something went wrong with creating one of the genrerics", e);
            //So let's try to create a plain object.
        }

        //Oops, something went wrong or there was just a wildcard.
        //The following lines gets all the loadable classes and picks one class.
        Set<Class<? extends Object>> asList = reflections.getSubTypesOf(Object.class);
        asList = getSetWithClassesThatOverridesASpecificMethod(asList, "equals", Object.class);
        asList = getSetWithClassesThatOverridesASpecificMethod(asList, "hashCode");
        return createRandomObjects(asList);
    }

    /**
    * This method creates a random object of a random class found with {@link org.unitils.objectvalidation.objectcreator.generator.WildCardTypeGenerator#getClassesOfSubtypes(sun.reflect.generics.reflectiveObjects.WildcardTypeImpl, java.util.Set)} or {@link org.unitils.objectvalidation.objectcreator.generator.WildCardTypeGenerator#getClassesOfSuperType(sun.reflect.generics.reflectiveObjects.WildcardTypeImpl, java.util.Set)}.
    *
    * @param classes: A set with a collection of classes.
    * @return {@link Object}
    */
   protected Object createRandomObjects(Set<Class<?>> classes) {
       List<Class<?>> tempClasses = new ArrayList<Class<?>>();
       //filter out all interfaces --> can't generate objects of interfaces unless there is a custom generator.
       //just to be sure try to remove all the interfaces.
       for (Class<?> clzz : classes) {
           if (!clzz.isInterface()) {
               tempClasses.add(clzz);
           }
       }

       if (tempClasses.isEmpty()) {
           //Apparantly there are only interfaces, just hope that the user has created a custom generator.
           tempClasses.addAll(classes);
       }

       Random random = new Random();

       Class<?> obj = tempClasses.get(random.nextInt(tempClasses.size()));

       return objenesisStd.getInstantiatorOf(obj).newInstance();
   }
    public static List<TreeNode> createNodes(Type type) {
        List<TreeNode> treeNodes = new ArrayList<TreeNode>();
        treeNodes.add(createTreenode(type));
        return treeNodes;
    }

    public static TreeNode createTreenode(Type type) {
        TypeWrapper wrapper = new TypeWrapper(type);
        if (!wrapper.isClassType()) {
            TreeNode node = new TreeNode(type.getClass());
            node.setType(type);
            return node;

        } else {
            TreeNode node = new TreeNode((Class<?>) type);
            node.setType(type);
            return node;
        }

    }


    public static TreeNode createTreenode(Field field) {
        if (!field.getGenericType().getClass().equals(Class.class)) {
            return createTreenode(field.getGenericType());

        } else {
            return createTreenode(field.getType());
        }
    }
    public static List<TreeNode> createTreenodes(Field field) {
        List<TreeNode> list = new ArrayList<TreeNode>();
        list.add(createTreenode(field));
        return list;
    }

}
