package org.unitils.objectvalidation.rulescollection;

import java.util.ArrayList;
import java.util.List;

import org.unitils.objectvalidation.EqualsHashCodeValidator;
import org.unitils.objectvalidation.ObjectCloner;
import org.unitils.objectvalidation.ObjectCreator;
import org.unitils.objectvalidation.ObjectValidationRulesCollection;
import org.unitils.objectvalidation.Rule;
import org.unitils.objectvalidation.cloner.ObjectClonerFactory;
import org.unitils.objectvalidation.objectcreator.ObjectCreatorFactory;
import org.unitils.objectvalidation.rules.ClassMustBeSerializableRule;
import org.unitils.objectvalidation.rules.EqualsComplientRule;
import org.unitils.objectvalidation.rules.GetterAndSetterComplientRule;
import org.unitils.objectvalidation.rules.HashCodeComplientRule;
import org.unitils.objectvalidation.rules.PublicEmptyConstructorRule;


/**
 * A set of default rules for the class validation.
 * Checks that :
 * 1. The equals is complient with the Sun rules : {@link EqualsComplientRule}
 * 2. The hashCode is complient with the Sun rules : {@link HashCodeComplientRule}
 * 3. The getters and setters are complient with the Sun rules : {@link GetterAndSetterComplientRule}
 * 4. The class is serializable and contains the serialVersionUID field : {@link ClassMustBeSerializableRule}
 * 6. There is a public empty constructor : {@link PublicEmptyConstructorRule}
 * 
 * @author Matthieu Mestrez
 * @since Oct 10, 2013
 */
public class SunBeanRules implements ObjectValidationRulesCollection {

    
    protected ObjectCreator objectCreator;
    protected ObjectCloner objectCloner;

    public SunBeanRules() {
        objectCreator = ObjectCreatorFactory.createDefaultObjectCreator();
        objectCloner = ObjectClonerFactory.createDefaultCloner();

    }
    
    @Override
    public List<Rule> getRules() {
        List<Rule> rules = new ArrayList<Rule>();
        
        rules.add(new EqualsComplientRule(objectCreator, objectCloner));
        rules.add(new HashCodeComplientRule(objectCreator, objectCloner));
        rules.add(new GetterAndSetterComplientRule(objectCreator, objectCloner));
        rules.add(new ClassMustBeSerializableRule());
        rules.add(new PublicEmptyConstructorRule());
        
        return rules;
    }

    @Override
    public EqualsHashCodeValidator getEqualsHashCodeValidator() {
        return new EqualsHashCodeValidator(objectCreator, objectCloner);
    }
        
}
